!DEC$ FREEFORM
!=============================================================================
! Interface to element routine for Abaqus
! (employed in Monolithic FE^2)
! Nils Lange, Geralf Huetter, Bjoern Kiefer
!   Nils.Lange@imfd.tu-freiberg.de, Geralf.Huetter@imfd.tu-freiberg.de, 
!   Bjoern.Kiefer@imfd.tu-freiberg.de
! distributed under CC BY-NC-SA 4.0 license
! (https://creativecommons.org/licenses/by-nc-sa/4.0/)
! Reference: 
!   N. Lange, G. Huetter, B. Kiefer: "An efficient monolithic solution scheme for FE2 problems",
!   https://arxiv.org/abs/2101.01802
!
! Further information on the implementation, structure of the source code,
! examples and tutorials can be found in the file doc/*.pdf
! 
! Description: Interface UEL routine adapted from example element of UELlib as a 
!   modular assembly of shape functions, quadrature rule and so on
!   - axi- and spheri-symmetric elements removed
!   - added: GET_n_STATEV_elem returns number of STATEVs of each element in dependance
!     of the element type (JTYPE) and the properties (PROPS)
!================================================================================

INCLUDE "ABQinterface.f90"
INCLUDE "Math.f90"
INCLUDE "UEL_lib.f90"
INCLUDE "UXMAT.f95"

MODULE ExtendedUELModule
    USE ABQINTERFACE
    TYPE UELextension
        REAL(KIND=AbqRK),DIMENSION(:,:),POINTER:: B_Matrices
        REAL(KIND=AbqRK),DIMENSION(:,:),POINTER:: F_int_ip
        REAL(KIND=AbqRK):: STRESS33 !Stress in thickness direction (plane strain)
        REAL(KIND=AbqRK),DIMENSION(:),POINTER:: IP_Weights
        INTEGER,DIMENSION(:,:),POINTER:: integration_points
        !true if training data is to be returned, false if trained data for the B-Matrix and the
        !integration points is to be used
        LOGICAL:: training
    END TYPE UELextension
END MODULE ExtendedUELModule

MODULE CXU2D6PlaneStrain
  USE ABQINTERFACE
  USE ShapeFunc2D_Triangle_Quad, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Triangle3GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D6PlaneStrain

MODULE CXU2D6PlaneStress
  USE ABQINTERFACE
  USE ShapeFunc2D_Triangle_Quad, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Triangle3GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D6PlaneStress

MODULE CXU2D4PlaneStrain
  USE ShapeFunc2D_Square_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square4GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D4PlaneStrain

MODULE CXU2D4PlaneStrainReduced
  USE ShapeFunc2D_Square_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square1GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D4PlaneStrainReduced

MODULE CXU2D8PlaneStrain
  USE ShapeFunc2D_Square_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square9GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D8PlaneStrain

MODULE CXU2D8PlaneStrainReduced
  USE ShapeFunc2D_Square_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square4GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D8PlaneStrainReduced

MODULE CXU2D4PlaneStress
  USE ShapeFunc2D_Square_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square4GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D4PlaneStress

MODULE CXU2D4PlaneStressReduced
  USE ShapeFunc2D_Square_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square1GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D4PlaneStressReduced

MODULE CXU2D8PlaneStress
  USE ShapeFunc2D_Square_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square9GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D8PlaneStress

MODULE CXU2D8PlaneStressReduced
  USE ShapeFunc2D_Square_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Square4GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU2D8PlaneStressReduced

MODULE CXU3D8
  USE ShapeFunc3D_Hex_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices3D, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr3D_Hex8GP ! For linear shape function 8 integration points provided
                      ! by Integr3D_HexRed module yield full integration scheme
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU3D8

MODULE CXU3D20
  USE ShapeFunc3D_Hex_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices3D, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr3D_Hex27GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU3D20

MODULE CXU3D20Reduced
  USE ShapeFunc3D_Hex_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices3D, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr3D_Hex8GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE CXU3D20Reduced

MODULE CXU3D4
  USE ShapeFunc3D_Tetrahed_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices3D, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr3D_Tetrahed1GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE

MODULE CXU3D10
  USE ShapeFunc3D_Tetrahed_Quadr, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices3D, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr3D_Tetrahed4GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE

MODULE CXU2D3PlaneStrain
  USE ShapeFunc2D_Triangle_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStrain, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Triangle1GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE

MODULE CXU2D3PlaneStress
  USE ShapeFunc2D_Triangle_Lin, ONLY: ShapeFunc, ShapeFuncDeriv, NDIM, NNODES
  USE BMatrices2DPlaneStress, ONLY: BMatTensSym, NDI, NSHR, SYM, PreFactor,NTENS,BMatTensSym_SFG
  USE BMatricesScalar, ONLY: BMatScal
  USE Integr2D_Triangle1GP
  USE UXMATModule, UMAT=>UXMAT
  IMPLICIT NONE
  PRIVATE
  PUBLIC :: UEL,NTENS,NGP,NDI
  CONTAINS
    INCLUDE "UelCXU.f95"
END MODULE

! Main routine with choice of element by number

SUBROUTINE UEL(RHS,AMATRX,SVARS,ENERGY,NDOFEL,NRHS,NSVARS,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC, &
               JELEM,PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)

  USE ExtendedUELModule !needed for MonolithFE2 in training or hyper ROM mode
  USE ABQinterface
  USE CXU2D4PlaneStrain, ONLY: UEL1 => UEL
  USE CXU2D8PlaneStrain, ONLY: UEL2 => UEL
  USE CXU2D4PlaneStress, ONLY: UEL3 => UEL
  USE CXU3D8, ONLY: UEL4 => UEL
  USE CXU3D20, ONLY: UEL5 => UEL
  USE CXU2D4PlaneStrainReduced, ONLY: UEL6 => UEL
  USE CXU2D8PlaneStrainReduced, ONLY: UEL7 => UEL
  USE CXU2D4PlaneStressReduced, ONLY: UEL8 => UEL
  USE CXU2D8PlaneStress, ONLY: UEL9 => UEL
  USE CXU2D8PlaneStressReduced, ONLY: UEL10 => UEL
  USE CXU3D20Reduced, ONLY: UEL11 => UEL
  USE CXU2D3PlaneStrain, ONLY: UEL12 => UEL
  USE CXU3D4, ONLY: UEL13 => UEL
  USE CXU3D10, ONLY: UEL14 => UEL
  USE CXU2D3PlaneStress, ONLY: UEL24=>UEL
  USE CXU2D6PlaneStrain, ONLY: UEL29 => UEL
  USE CXU2D6PlaneStress, ONLY: UEL30 => UEL
  
  IMPLICIT NONE
  
  !provide an explicit interface
  INTEGER(KIND=AbqIK):: NDOFEL,NRHS,NSVARS,NPROPS,NJPROP,NNODE,NDLOAD,NPREDF,MLVARX,MDLOAD,MCRD
  INTEGER(KIND=AbqIK):: JTYPE,JELEM,KSTEP,KINC
  INTEGER(KIND=AbqIK),DIMENSION(*):: LFLAGS
  INTEGER(KIND=AbqIK),DIMENSION(NJPROP):: JPROPS
  INTEGER(KIND=AbqIK),DIMENSION(MDLOAD,NDLOAD):: JDLTYP
  REAL(KIND=AbqRK):: DTIME,PNEWDT,PERIOD
  REAL(KIND=AbqRK),DIMENSION(2):: TIME
  REAL(KIND=AbqRK),DIMENSION(MLVARX,NRHS):: RHS
  REAL(KIND=AbqRK),DIMENSION(NDOFEL,NDOFEL):: AMATRX
  REAL(KIND=AbqRK),DIMENSION(NSVARS):: SVARS
  REAL(KIND=AbqRK),DIMENSION(8):: ENERGY
  REAL(KIND=AbqRK),DIMENSION(NPROPS):: PROPS
  REAL(KIND=AbqRK),DIMENSION(MCRD,NNODE):: COORDS
  REAL(KIND=AbqRK),DIMENSION(NDOFEL):: U,V,A
  REAL(KIND=AbqRK),DIMENSION(MLVARX,NRHS):: DU
  REAL(KIND=AbqRK),DIMENSION(*):: PARAMS
  REAL(KIND=AbqRK),DIMENSION(MDLOAD,*):: ADLMAG
  REAL(KIND=AbqRK),DIMENSION(MDLOAD,*):: DDLMAG
  REAL(KIND=AbqRK),DIMENSION(2,NPREDF,NNODE):: PREDEF
  !optional argument
  TYPE(UELextension),OPTIONAL:: extension
  LOGICAL:: extension_present
  
  !find out if extension is present, unfortunately PRESENT(extension) isn't enough,
  !since it seems to give true sometimes, even if it isn't present actually...
  IF (NJPROP>0) THEN
    IF (JPROPS(1)==1) THEN
        extension_present=.TRUE.
    ELSE
        extension_present=.FALSE.
    END IF
  ELSE
    extension_present=.FALSE.
  END IF

  IF (extension_present) THEN
    SELECT CASE(JTYPE)
        CASE (2001)
        CALL UEL1(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2002)
        CALL UEL2(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2003)
        CALL UEL3(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2004)
        CALL UEL4(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2005)
        CALL UEL5(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2006)
        CALL UEL6(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2007)
        CALL UEL7(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2008)
        CALL UEL8(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2009)
        CALL UEL9(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2010)
        CALL UEL10(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2011)
        CALL UEL11(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2012)
        CALL UEL12(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2013)
        CALL UEL13(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2014)
        CALL UEL14(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2024)
        CALL UEL24(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2029)
        CALL UEL29(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE (2030)
        CALL UEL30(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD,extension)
        CASE DEFAULT
        CALL STDB_ABQERR(-3,"Error: UEL element routine for element U%I not defined", JTYPE, 0.0, " ")
    END SELECT
  ELSE
    SELECT CASE(JTYPE)
        CASE (2001)
        CALL UEL1(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2002)
        CALL UEL2(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2003)
        CALL UEL3(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2004)
        CALL UEL4(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2005)
        CALL UEL5(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2006)
        CALL UEL6(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2007)
        CALL UEL7(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2008)
        CALL UEL8(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2009)
        CALL UEL9(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2010)
        CALL UEL10(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2011)
        CALL UEL11(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2012)
        CALL UEL12(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2013)
        CALL UEL13(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2014)
        CALL UEL14(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2024)
        CALL UEL24(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2029)
        CALL UEL29(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE (2030)
        CALL UEL30(RHS,AMATRX,SVARS,ENERGY,NDOFEL,PROPS,NPROPS,COORDS,MCRD,NNODE,U,DU,V,A,JTYPE,TIME,DTIME,KSTEP,KINC,JELEM, &
                    PARAMS,NDLOAD,JDLTYP,ADLMAG,PREDEF,NPREDF,LFLAGS,MLVARX,DDLMAG,MDLOAD,PNEWDT,JPROPS,NJPROP,PERIOD)
        CASE DEFAULT
        CALL STDB_ABQERR(-3,"Error: UEL element routine for element U%I not defined", JTYPE, 0.0, " ")
    END SELECT
  END IF
END SUBROUTINE

SUBROUTINE GET_n_STATEV_elem(n_STATEV_elem,JTYPE,PROPS,NPROPS,NTENS)
!returns number of necessary STATEVS per Element, dependend on the used Element-
!type and the PROPS

  USE CXU2D3PlaneStrain, ONLY: NTENS_2D3=>NTENS, NGP_2D3=>NGP, NDI_2D3=>NDI
  USE CXU2D4PlaneStrain, ONLY: NTENS_2D4=>NTENS, NGP_2D4=>NGP, NDI_2D4=>NDI
  USE CXU3D4, ONLY: NTENS_3D4=>NTENS, NGP_3D4=>NGP, NDI_3D4=>NDI
  USE CXU3D8, ONLY: NTENS_3D8=>NTENS, NGP_3D8=>NGP, NDI_3D8=>NDI
  USE CXU2D4PlaneStress, ONLY: NTENS_2D4_PS=>NTENS, NGP_2D4_PS=>NGP, NDI_2D4_PS=>NDI
  USE CXU2D3PlaneStress, ONLY: NTENS_2D3_PS=>NTENS, NGP_2D3_PS=>NGP, NDI_2D3_PS=>NDI
  USE CXU2D8PlaneStrain, ONLY: NTENS_2D8=>NTENS, NGP_2D8=>NGP, NDI_2D8=>NDI
  USE CXU3D20, ONLY: NTENS_3D20=>NTENS, NGP_3D20=>NGP, NDI_3D20=>NDI
  USE CXU2D8PlaneStrainReduced, ONLY: NTENS_2D8_R=>NTENS, NGP_2D8_R=>NGP, NDI_2D8_R=>NDI
  USE CXU3D20Reduced, ONLY: NTENS_3D20_R=>NTENS, NGP_3D20_R=>NGP, NDI_3D20_R=>NDI
  USE CXU2D4PlaneStrainReduced, ONLY: NTENS_2D4_R=>NTENS, NGP_2D4_R=>NGP, NDI_2D4_R=>NDI
  USE CXU2D4PlaneStressReduced, ONLY: NTENS_2D4_PS_R=>NTENS, NGP_2D4_PS_R=>NGP, NDI_2D4_PS_R=>NDI
  USE CXU2D8PlaneStress, ONLY: NTENS_2D8_PS=>NTENS, NGP_2D8_PS=>NGP, NDI_2D8_PS=>NDI
  USE CXU2D8PlaneStressReduced, ONLY: NTENS_2D8_PS_R=>NTENS, NGP_2D8_PS_R=>NGP, NDI_2D8_PS_R=>NDI
  USE CXU3D10, ONLY: NTENS_3D10=>NTENS, NGP_3D10=>NGP, NDI_3D10=>NDI
  USE CXU2D6PlaneStrain, ONLY: NTENS_2D6=>NTENS, NGP_2D6=>NGP, NDI_2D6=>NDI
  USE CXU2D6PlaneStress, ONLY: NTENS_2D6_PS=>NTENS, NGP_2D6_PS=>NGP, NDI_2D6_PS=>NDI

  USE UXMATModule,ONLY: GetNSTATV
  USE ABQINTERFACE
  
  IMPLICIT NONE
  
  INTEGER,INTENT(OUT):: n_STATEV_elem,NTENS
  INTEGER,INTENT(IN):: JTYPE, NPROPS
  REAL(KIND=AbqRK),INTENT(IN),DIMENSION(NPROPS):: PROPS
      
  SELECT CASE( JTYPE )
    CASE (2001)
        n_STATEV_elem=NGP_2D4*(GetNSTATV(NTENS_2D4,NDI_2D4,PROPS)+NTENS_2D4)
        NTENS=NTENS_2D4
    CASE (2002)
        n_STATEV_elem=NGP_2D8*(GetNSTATV(NTENS_2D8,NDI_2D8,PROPS)+NTENS_2D8)
        NTENS=NTENS_2D8
    CASE (2003)
        n_STATEV_elem=NGP_2D4_PS*(GetNSTATV(NTENS_2D4_PS,NDI_2D4_PS,PROPS)+NTENS_2D4_PS)
        NTENS=NTENS_2D4_PS
    CASE (2004)
        n_STATEV_elem=NGP_3D8*(GetNSTATV(NTENS_3D8,NDI_3D8,PROPS)+NTENS_3D8)
        NTENS=NTENS_3D8
    CASE (2005)
        n_STATEV_elem=NGP_3D20*(GetNSTATV(NTENS_3D20,NDI_3D20,PROPS)+NTENS_3D20)
        NTENS=NTENS_3D20
    CASE (2006)
        n_STATEV_elem=NGP_2D4_R*(GetNSTATV(NTENS_2D4_R,NDI_2D4_R,PROPS)+NTENS_2D4_R)
        NTENS=NTENS_2D4_R
    CASE (2007)
        n_STATEV_elem=NGP_2D8_R*(GetNSTATV(NTENS_2D8_R,NDI_2D8_R,PROPS)+NTENS_2D8_R)
        NTENS=NTENS_2D8_R
    CASE (2008)
        n_STATEV_elem=NGP_2D4_PS_R*(GetNSTATV(NTENS_2D4_PS_R,NDI_2D4_PS_R,PROPS)+NTENS_2D4_PS_R)
        NTENS=NTENS_2D4_PS_R
    CASE (2009)
        n_STATEV_elem=NGP_2D8_PS*(GetNSTATV(NTENS_2D8_PS,NDI_2D8_PS,PROPS)+NTENS_2D8_PS)
        NTENS=NTENS_2D8_PS
    CASE (2010)
        n_STATEV_elem=NGP_2D8_PS_R*(GetNSTATV(NTENS_2D8_PS_R,NDI_2D8_PS_R,PROPS)+NTENS_2D8_PS_R)
        NTENS=NTENS_2D8_PS_R
    CASE (2011)
        n_STATEV_elem=NGP_3D20_R*(GetNSTATV(NTENS_3D20_R,NDI_3D20_R,PROPS)+NTENS_3D20_R)
        NTENS=NTENS_3D20_R
    CASE (2012)
        n_STATEV_elem=NGP_2D3*(GetNSTATV(NTENS_2D3,NDI_2D3,PROPS)+NTENS_2D3)
        NTENS=NTENS_2D3
    CASE (2013)
        n_STATEV_elem=NGP_3D4*(GetNSTATV(NTENS_3D4,NDI_3D4,PROPS)+NTENS_3D4)
        NTENS=NTENS_3D4
    CASE (2014)
        n_STATEV_elem=NGP_3D10*(GetNSTATV(NTENS_3D10,NDI_3D10,PROPS)+NTENS_3D10)
        NTENS=NTENS_3D10
    CASE (2024)
        n_STATEV_elem=NGP_2D3_PS*(GetNSTATV(NTENS_2D3_PS,NDI_2D3_PS,PROPS)+NTENS_2D3_PS)
        NTENS=NTENS_2D3_PS
    CASE (2029)
        n_STATEV_elem=NGP_2D6*(GetNSTATV(NTENS_2D6,NDI_2D6,PROPS)+NTENS_2D6)
        NTENS=NTENS_2D6
    CASE (2030)
        n_STATEV_elem=NGP_2D6_PS*(GetNSTATV(NTENS_2D6_PS,NDI_2D6_PS,PROPS)+NTENS_2D6_PS)
        NTENS=NTENS_2D6_PS
    CASE DEFAULT
      CALL STDB_ABQERR(-3,"Error: UEL element routine for element U%I not defined", JTYPE, 0.0, " ")
  END SELECT
    
END SUBROUTINE GET_n_STATEV_elem
