C=======================================================================
C TU Bergakademie Freiberg
C Faculty of Mechanical, Process and Energy Engineering
C Institute for Mechanics and Fluid Dynamics (IMFD)
************************************************************************
C Implementation of non-local GTN-Model for ductile damage in ABAQUS
C Version 1.0
C Date 17 May 2022
C Author: Andreas Seupel, Geralf Hütter, Rinh Pham, Omar El Khatib,
C         Bjoern Kiefer
************************************************************************
C Corresponding author:
C     andreas.seupel@imfd.tu-freiberg.de (A.Seupel)
C     geralf.huetter@imfd.tu-freiberg.de (G.Hütter)
C     dinh-rinh.pham1@imfd.tu-freiberg.de (R. Pham)
C Reference:
C G.Hütter, T.Linse, U.Mühlich and M.Kuna: Simulation of Ductile Crack 
C           Initiation and Propagation by means of a Non-local GTN-model 
C           under Small-Scale Yielding, International Journal of Solids 
C           and Structures, 50 (2013), 662--671.
C A.Seupel, G.Hütter, and M.~Kuna: On the identification and uniqueness 
C           of constitutive parameters for a non-local GTN-model, 
C           Engineering Fracture Mechanics, 229 (2020), 106817.
C R.D. Pham, O. El Khatib, A. Seupel, G. Hütter and B. Kiefer: 
C           Non-iterative determination of the parameters for the Gurson
C           model from two standard tests, DVM-Bericht 254(2022), 43-52
C O.El Khatib, G.Hütter, D.R.Pham, A.~Seupel, M.~Kuna and B.~Kiefer: 
C           A non-iterative parameter calibration procedure for the 
C           GTN model based on standardized experiments,"in preparation".
C***********************************************************************
C=======================================================================
C***********************************************************************
C Integrate the module to explicit variable declaration 
C and interface definition
C 
      MODULE Abaqus_Interface
            INCLUDE 'ABA_PARAM.INC'
            PRIVATE
            INTEGER,PARAMETER,PUBLIC::rk=KIND(r),intk=KIND(i)
      END MODULE
C***********************************************************************
C=======================================================================
C***********************************************************************
C Variables for UMAT Subroutine in ABAQUS
C Reference to ABAQUS documentation for more details
C ----------------------------------------------------------------------
C Variables to be defined
C DDSDDE    Jacobian matrix of the constitutive model
C STRESS    Stress tensor at the beginning of the increment
C                 and must be updated at the end of the increment
C STATEV    An array containing the solution-dependent state variables
C SSE       Specific elastic strain energy
C SPD       Plastic dissipation
C SCD       Creep dissipation
C RPL       Volumetric heat generation per unit time
C DDSDDT    Variation of the stress increments with respect 
C                 to the temperature
C DRPLDE    Variation of RPL with respect to the strain increments
C DRPLDT    Variation of RPL with respect to the temperature
C ----------------------------------------------------------------------
C Variables that can be updated
C PNEWDT    Ratio of suggested new time increment
C ----------------------------------------------------------------------
C Variables passed in for information
C STRAN     An array containing the total strains at the beginning 
C                 of the increment
C DSTRAN    Array of strain increments
C TIME(1)   Value of step time at the beginning of the current increment
C TIME(2)   Value of total time at the beginning of the current increment
C DTIME     Time increment
C TEMP      Temperature at the start of the increment
C DTEMP     Increment of temperature
C PREDEF    Array of interpolated values of predefined field variables
C DPRED     Array of increments of predefined field variables
C CMNAME    User-defined material name, left justified
C NDI       Number of direct stress components
C NSHR      Number of engineering shear stress components
C NTENS     Size of the stress or strain component array (NDI + NSHR)
C NSTATV    Number of solution-dependent state variables
C PROPS     User-specified array of material constants 
C NPROPS    User-defined number of material constants
C COORDS    An array containing the coordinates
C DROT(3,3) Rotation increment matrix
C CELENT    Characteristic element length
C DFGRD0(3,3)     Array containing the deformation gradient 
C                 at the beginning of the increment
C DFGRD1(3,3)     Array containing the deformation gradient 
C                 at the end of the increment
C NOEL      Element number
C NPT       Integration point number
C LAYER     Layer number 
C KSPT      Section point number within the current layer
C JSTEP(1)  Step number
C JSTEP(2)  Procedure type key
C JSTEP(3)  1 if NLGEOM=YES for the current step; 0 otherwise
C JSTEP(4)  1 if current step is a linear perturbation procedure; 
C                 0 otherwise
C KINC      Increment number
C-----------------------------------------------------------------------
C A Abaqus standard subroutine
C SUBROUTINE UMAT is the main part of the code. Others Subroutines are 
C therefore will be called directly or indirectly from SUBROUTINE UMAT.
C
      SUBROUTINE UMAT(STRESS,STATEV,DDSDDE,SSE,SPD,SCD,RPL,DDSDDT,
     1               DRPLDE,DRPLDT,STRAN,DSTRAN,TIME,DTIME,TEMP,DTEMP,
     2               PREDEF,DPRED,CMNAME,NDI,NSHR,NTENS,NSTATV,PROPS,
     3               NPROPS,COORDS,DROT,PNEWDT,CELENT,DFGRD0,DFGRD1,
     4               NOEL,NPT,LAYER,KSPT,JSTEP,KINC)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Standard variables for Umat subroutine in ABAQUS
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(INOUT)      ::STRESS
      REAL(KIND=rk),DIMENSION(NSTATV),INTENT(INOUT)     ::STATEV
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(OUT)  ::DDSDDE
      REAL(KIND=rk),INTENT(INOUT)::SSE,SPD,SCD
      REAL(KIND=rk),INTENT(OUT)  ::RPL
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(OUT) ::DDSDDT,DRPLDE
      REAL(KIND=rk),INTENT(OUT)  ::DRPLDT
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN)  ::STRAN,DSTRAN
      REAL(KIND=rk),DIMENSION(2),INTENT(IN)      ::TIME
      REAL(KIND=rk),INTENT(IN)   ::DTIME,TEMP,DTEMP,PREDEF,DPRED
      CHARACTER(80)::CMNAME
      INTEGER(KIND=intk),INTENT(IN) ::NDI,NSHR,NTENS,NSTATV
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN) ::PROPS
      INTEGER(KIND=intk),INTENT(IN) ::NPROPS
      REAL(KIND=rk),DIMENSION(3),INTENT(IN)      ::COORDS
      REAL(KIND=rk),DIMENSION(3,3),INTENT(IN)    ::DROT
      REAL(KIND=rk),INTENT(INOUT)                ::PNEWDT
      REAL(KIND=rk),INTENT(IN)                   ::CELENT
      REAL(KIND=rk),DIMENSION(3,3),INTENT(IN)    ::DFGRD0,DFGRD1
      INTEGER(KIND=intk),INTENT(IN) ::NOEL,NPT,LAYER,KSPT
      INTEGER(KIND=intk),DIMENSION(4),INTENT(IN)    ::JSTEP
      INTEGER(KIND=intk),INTENT(IN)                 ::KINC
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Local variables declaration
      INTEGER(KIND=intk) ::PNEW,SUBSTEPCONTROL
      REAL(KIND=rk) ::F_LOK,DELTA_F_LOK,DELTA_E_EQ_GES,DEPS_VOL
      REAL(KIND=rk),DIMENSION(NTENS,1)  ::DEVIATOR_EPS_MATRIX,
     1                               DJAC_DEPS,STRESS_MATRIX
      REAL(KIND=rk),DIMENSION(NTENS)    ::DEVIATOR_EPS,STRESS_ALT
      INTEGER(KIND=intk),DIMENSION(1)  ::MAXIMALWERT
      INTEGER(KIND=intk)               ::i,j,LOKAL_NLOKAL
      REAL(KIND=rk)::JAC,EPSILON_MISES,RR,DAMAGE,FCUMSCHALTEN,
     1       T,THETA,DELTATHETA,DELTATHETAMIN,I1T,J2T,
     2       TAU_MISES_T,TAU_HYD_T,DAMAGEOUT,TAU_MISES,TAU_HYD,COS_H 
      REAL(KIND=rk)::R,FSTAR,HH,DFSTAR_DFEFF,
     1       FLIESSEN,FLIESSENTR,DELTA_LAMBDA_PL,DELTA_LAMBDA_TR,
     2       DELTA_LAMBDA_PLV,DELTA_RR,DELTA_E_EQ_NL,DTIME_INK,
     3       DELTA_D,DELTA_NUKLEATION
      REAL(KIND=rk),DIMENSION(NTENS)    ::EPS,KRONECKER,TAU_ROT,
     1                                    EPSGES,DEVIATORT,N_TR,TAU_T
      REAL(KIND=rk),DIMENSION(2)           ::STARTDELTA
      REAL(KIND=rk),DIMENSION(NTENS,NTENS) ::CMATRIX,NORMMATRIX,
     1                                    IMATRIX,IDEV
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Check number of user material paramters
      IF (KINC==1) THEN
         IF (NPROPS==13) THEN
          IF (NOEL==1 .AND. NPT==1) THEN
            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
            CALL STDB_ABQERR(1,'STRAIN HARDENING DEFINED THROUGH '//
     1                'POWER LAW WHICH IS SPECIFIED BY 2 PARAMETERS:'//
     2                   'INITIAL YIELD STRESS AND HARDENING EXPONENT',
     3                    1,1.0_rk,'')
          END IF
         ELSEIF (NPROPS>13) THEN
          IF (NOEL==1 .AND. NPT==1) THEN
            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
            CALL STDB_ABQERR(1,'STRAIN HARDENING DEFINED THROUGH '//
     1             'TABULAR FORM',1,1.0_rk,'')
          END IF
         ELSE
            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
            CALL STDB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO WRONG'//
     1           ' INPUT, KEYWORD *MATERIAL: NUMBER OF PROPS TO BE'//
     2           ' DEFINED: 13 or > 13',1,1.0_rk,'')
         END IF
C     Check option for local or non-local GTN-Model
         IF (DAMAGE_CASE<0.5_rk) THEN
            LOKAL_NLOKAL    =0
         ELSE
            LOKAL_NLOKAL    =1
         END IF
C     Check dimension ( 2D vs 3D), if 2D, then check whether it is plain
C     strain or plain stress. 
C     UMAT does not support "plane stress" problem
         IF (NTENS==6) THEN
           IF (NSTATV==11) THEN
           ELSE
            PNEW    =1
               CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
               CALL STDB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO '//
     1                     'WRONG INPUT, KEYWORD *MATERIAL: NUMBER OF'//
     2                     ' STATEV (DEPVAR) ALLOWED FOR ANALYSIS: 11',
     3                     1,1.0_rk,'')
           END IF
         ELSEIF (NTENS==4) THEN
           IF (NSTATV==11) THEN
           ELSE
              PNEW    =1
              CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
              CALL STDB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO '//
     1                     'WRONG INPUT, KEYWORD *MATERIAL: NUMBER OF'//
     2                     ' STATEV (DEPVAR) ALLOWED FOR ANALYSIS: 11',
     3                     1,1.0_rk,'')
           END IF
C     Check if plane stress is given, flag up an ERROR!!!
         ELSEIF (NTENS .GT. 3) THEN
            PNEW    =1
            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
            CALL STDB_ABQERR(-3,'SIMULATION INTERRUPTED DUE',
     1                   'TO WRONG USAGE, PLANE STRESS STATE IS NOT',
     2                   'SUPPORTED',1,1.0_rk,'')
         END IF
      END IF
C     Caculation bulk modulus and shear modulus
      K = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G = E/(2.0_rk*(1.0_rk+NU))
C     Lame paramter  - Lambda
      LAMBDA      = K - 2.0_rk/3.0_rk*G
C     Local control paramter
      PNEW        = 0
C     Define Kronecker delta as a vector
      KRONECKER   =0.0_rk
      DO i=1,NDI,1
        KRONECKER(i)    =1.0_rk
      END DO
C     Define the CMATRIX for elastic response
      CMATRIX     =0.0_rk
C     CMATRIX is created only for 2D-Plain strain or 3D Problem
      IF (NTENS==3) THEN
         DO i=1,NTENS,1
               DO j=1,NTENS,1
                  IF (i<=NDI .AND. j<=NDI .AND. i/=j) THEN
                     CMATRIX(i,j) = (E/(1.0_rk-NU**2)*(NU))
                  END IF
               END DO
               IF (i<=NDI) THEN
                  CMATRIX(i,i)      = E/(1.0_rk-NU**2)
               ELSE
                  CMATRIX(i,i)      = G
               END IF
         END DO
      ELSE
         DO i=1,NTENS,1
            DO j=1,NTENS,1
               IF (i<=NDI .AND. j<=NDI) THEN
                  CMATRIX(i,j)      = LAMBDA
               END IF
            END DO
            IF (i<=NDI) THEN
               CMATRIX(i,i)      = LAMBDA+2.0_rk*G
            ELSE
               CMATRIX(i,i)      = G
            END IF
         END DO
      END IF
C     Create 'NORMMATRIX' for caculating scalar product from matrixes
      NORMMATRIX  =0.0_rk
C     NORMMATRIX is created only for 2D-Plain strain or 3D Problem
      IF (NTENS==3) THEN
         NORMMATRIX(1,1)     =2.0_rk
         NORMMATRIX(2,2)     =2.0_rk
         NORMMATRIX(1,2)     =1.0_rk
         NORMMATRIX(2,1)     =1.0_rk
         NORMMATRIX(3,3)     =2.0_rk
      ELSE
         DO i=1,NTENS,1
            IF (i<=NDI) THEN
               NORMMATRIX(i,i) = 1.0_rk
            ELSE
               NORMMATRIX(i,i) = 2.0_rk
            END IF
         END DO
      END IF
C     Create a unit tensor, 4th order
      IMATRIX     =0.0_rk
      DO i=1,NTENS,1
        IMATRIX(i,i)    =1.0_rk
      END DO
C     Create deviatoric unit tensor, 4th order
      CALL VECTOR_PRODUCT (NDI,NTENS,KRONECKER,KRONECKER,IDEV)
      IDEV    =IMATRIX-1.0_rk/3.0_rk*IDEV
C     Assgin value for internal state variables for firt increment
      IF (KINC==1 .AND. JSTEP(1)==1) THEN
            DO i=1,NSTATV,1
                STATEV(i)=0.0_rk
            END DO
            STATEV(8)   =F_C
            STATEV(10)  =F_0
      END IF
C     Convert stress into Kirchhoff-stress
      JAC                   = EXP(DOT_PRODUCT(STRAN+DSTRAN,KRONECKER))
      DJAC_DEPS(1:NTENS,1)  = JAC*KRONECKER(1:NTENS)

C     Assign strain increment in Abaqus to local variable
      EPS     =DSTRAN

C     Assign value for local internal variables
      EPSILON_MISES  =STATEV(1)
      RR             =STATEV(2)
      DAMAGE         =STATEV(4)
C
      FCUMSCHALTEN   =STATEV(6)
      F_LOK          =STATEV(10)

C     Assign stress value in Abaqus to local variable
      STRESS_ALT     =STRESS

C     Assign stress value in Abaqus to calculate trial stress
      TAU_ROT        =STRESS

C     Activate internal step control variables
      THETA           =0.0_rk
      DELTATHETA      =1.0_rk
      DELTATHETAMIN   =1.0_rk/10.0_rk
   
      SUBSTEPCONTROL  =1
   
      EPSGES          =EPS
      STARTDELTA      =0.0_rk

C Do-Loop for internal step control
      DO WHILE (SUBSTEPCONTROL==1 .AND. PNEW==0)
C     Calculate strain and time increment with respect to step control
         EPS                 =EPSGES*(THETA+DELTATHETA)
         DTIME_INK           =DTIME*(THETA+DELTATHETA)
         DELTA_E_EQ_NL       =DTEMP*(THETA+DELTATHETA)
C     Trial stress whether elastic or plastic response
         TAU_T               =TAU_ROT+MATMUL(CMATRIX,EPS)
C     Calculate values from trial stress
         I1T         =DOT_PRODUCT(TAU_T,KRONECKER)
         TAU_HYD_T       =I1T/3.0_rk
         DEVIATORT   =TAU_T-1.0_rk/3.0_rk*KRONECKER*I1T
         J2T         =1.0_rk/2.0_rk*DOT_PRODUCT(DEVIATORT,
     1                         MATMUL(NORMMATRIX,DEVIATORT))
         IF (J2T==0.0_rk) THEN
             TAU_MISES_T     =sqrt(3.0_rk*J2T)
             N_TR            =0.0_rk
         ELSE
             TAU_MISES_T     =sqrt(3.0_rk*J2T)
             N_TR            =3.0_rk/2.0_rk/TAU_MISES_T*DEVIATORT
         END IF
C     Deviatoric strain tensor
         DEVIATOR_EPS=MATMUL(IDEV,EPS)
         DEVIATOR_EPS_MATRIX(1:NTENS,1)  =DEVIATOR_EPS(1:NTENS)
C     Karush-Kuhn-Tucker condition for damage evolution
         IF (LOKAL_NLOKAL==0 ) THEN
C----->>>Local GTN Model
C        For local model delta_f_lok should be iterate as well, for 
C        that reason, it will be defined inside local subroutine
            DELTA_F_LOK    = 0.0_rk
         ELSE
C----->>>Non-local GTN Model
C        Karush-Kuhn-Tucker condition for damage evolution in nonlocal-model
            IF (STATEV(3)>TEMP+DELTA_E_EQ_NL) THEN
               DELTA_F_LOK =0.0_rk
               DEPS_VOL    =0.0_rk
            ELSE
               DELTA_F_LOK=1.0_rk-(1.0_rk-F_0)*EXP(-3.0_rk
     1                               *(TEMP+DELTA_E_EQ_NL))-STATEV(10)
               DEPS_VOL    =3.0_rk*(1.0_rk-STATEV(10)-DELTA_F_LOK)
            END IF
         END IF
C     Calculate current yield stress and Fstar for yield function
         CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR,STATEV(8),
     1             F_LOK+DELTA_F_LOK,R,FSTAR,HH,DFSTAR_DFEFF,
     2             EPSILON_MISES,DTIME,-1.0_rk)
C     Calculate the value of yield function
         COS_H       =COSH(Q_2*3.0_rk/2.0_rk*TAU_HYD_T/R)
         FLIESSENTR  =TAU_MISES_T**2/R**2+2.0_rk*Q_1*FSTAR*
     1                           COS_H-1.0_rk-Q_1**2*FSTAR**2
         FLIESSEN    =FLIESSENTR
C
C     Check material total damage under the current load
C
C     Check material response ( elastic vs elastoplastic)
C>>>>>Elastic response
         IF (SIGN(1.0_rk,FLIESSEN)<=0.0_rk) THEN
C----------> Local GTN-Model elastic response
            IF (LOKAL_NLOKAL==0) THEN
c!               write(*,*) 'Elastic regime for local model'
               DELTA_LAMBDA_PL         =0.0_rk
               DELTA_LAMBDA_PLV        =0.0_rk
               DELTA_RR                =0.0_rk
               DELTA_F_LOK             =0.0_rk
               DDSDDE                  =CMATRIX
               STRESS                  =TAU_T
               DAMAGEOUT               =DAMAGE
               DDSDDT                  =0.0_rk
               DRPLDE      =0.0_rk
               DRPLDT      =0.0_rk
C----------> Non-local GTN-Model elastic response
            ELSE
c!               write(*,*) 'Elastic regime'
               DELTA_LAMBDA_PL         =0.0_rk
               DELTA_LAMBDA_PLV        =0.0_rk
               DELTA_RR                =0.0_rk
               DDSDDE                  =CMATRIX
               STRESS                  =TAU_T
               DDSDDT                  =0.0_rk
               DRPLDE      =0.0_rk
               DRPLDT      =0.0_rk
C     End of If-condition for elastic response
            END IF
C
            PNEW                    =0
C>>>>>Elastoplastic response 
         ELSE
C----------> Local GTN-Model elastoplastic response
            IF (LOKAL_NLOKAL==0) THEN
C
               CALL PLASTICITY_MODULE(DELTA_E_EQ_GES,DTIME,NTENS,NPROPS,
     1                   NSTATV,STARTDELTA,TAU_T,DEVIATORT,N_TR,CMATRIX,
     2                   KRONECKER,IMATRIX,IDEV,NORMMATRIX,PROPS,STATEV,
     3                   I1T,J2T,TAU_MISES_T,TAU_HYD_T,STRESS,DDSDDE,
     4                   DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,DELTA_RR,
     5                   DELTA_F_LOK,DAMAGEOUT,DELTA_NUKLEATION,PNEW)
C
               IF (PNEW==0) THEN
C     Update the start value by using better start value
                  STARTDELTA(1)      =DELTA_LAMBDA_PL
                  STARTDELTA(2)      =DELTA_LAMBDA_PLV
C     Update the material tangent stiffness matrix
                  DDSDDE       =MATMUL(DDSDDE,CMATRIX)
                  DDSDDT       =0.0_rk
                  DRPLDE       =0.0_rk
                  DRPLDT       =0.0_rk
               ELSE
                  DDSDDE      =CMATRIX
                  DDSDDT      =0.0_rk
                  STRESS      =TAU_ROT
                  DAMAGEOUT   =DAMAGE
                  DELTA_F_LOK =0.0_rk
                  DRPLDE      =0.0_rk
                  DRPLDT      =0.0_rk
               END IF
C----------> Non-local GTN-Model elastoplastic response
            ELSE
               CALL PLASTICITY_MODULE_NL(DELTA_E_EQ_GES,DTIME_INK,
     1                  NTENS,NPROPS,NSTATV,STARTDELTA,TAU_T,DEVIATORT,
     2                  N_TR,CMATRIX,KRONECKER,IMATRIX,IDEV,NORMMATRIX,
     3                  PROPS,STATEV,I1T,J2T,TAU_MISES_T,
     4                  TAU_HYD_T,STRESS,DDSDDE,DDSDDT,DRPLDE,DRPLDT,
     5                  DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,DELTA_RR,
     6                  DELTA_F_LOK,DELTA_NUKLEATION,PNEW,CMNAME)
C
               IF (PNEW==0) THEN
C     Update the start value by using better start value
                  STARTDELTA(1)      =DELTA_LAMBDA_PL
                  STARTDELTA(2)      =DELTA_LAMBDA_PLV
C     Update the material tangent stiffness matrix
                  DDSDDE   =MATMUL(DDSDDE,CMATRIX)
                  DRPLDE   =MATMUL(CMATRIX,DRPLDE)
                  DDSDDT   =DDSDDT*DEPS_VOL
               ELSE
                  DDSDDE      =CMATRIX
                  DDSDDT      =0.0_rk
                  DRPLDE      =0.0_rk
                  DRPLDT      =0.0_rk
                  STRESS      =TAU_ROT
               END IF
C     End of If-condition for elastoplastic response
            END IF
C
         END IF 
C>>>>>End of checking material response
C
C     Increment control for non-convergence material point
         IF (THETA+DELTATHETA-1.0_rk==0.0_rk .AND. PNEW==0) THEN
C
               SUBSTEPCONTROL      =0
C
         ELSE
               IF (PNEW==1) THEN
  
                   DELTATHETA     = DELTATHETA/2.0_rk
  
                   IF (DELTATHETA<DELTATHETAMIN) THEN
                   ELSE
                       PNEW    =0
                   END IF
  
               ELSE
                   THETA                   =THETA+DELTATHETA
                   DELTATHETA              =DELTATHETA*1.5_rk
  
                   IF (DELTATHETA+THETA - 1.0_rk>=0.0_rk) THEN
                   DELTATHETA          =1.0_rk-THETA
                   END IF
  
               END IF
C
         END IF
C     End of increment control for non-convergence material point
C
      END DO
C End of Do-Loop for internal step control

C     Test on NAN
      IF (PNEW==0) THEN
         DO i=1,NTENS,1
            IF (STRESS(i).NE. STRESS(i)) THEN
                IF (PNEW==0) THEN
                    PNEW=1
                    CALL STDB_ABQERR(1,'NAN',1,1.0_rk,'')
                    CALL STDB_ABQERR(1,'STRESS',1,1.0_rk,'')
                    write(*,*) STRESS
                END IF
            END IF
         END DO
      END IF

      IF (PNEW==0) THEN
         IF (DELTA_LAMBDA_PL .NE. DELTA_LAMBDA_PL) THEN
            PNEW=1
            CALL STDB_ABQERR(1,'NAN',1,1.0_rk,'')
            CALL STDB_ABQERR(1,'DELTA_LAMBDA_PL',1,1.0_rk,'')
         END IF
      END IF

      IF (PNEW==0) THEN
         IF (DELTA_LAMBDA_TR .NE. DELTA_LAMBDA_TR) THEN
            PNEW=1
            CALL STDB_ABQERR(1,'NAN',1,1.0_rk,'')
            CALL STDB_ABQERR(1,'DELTA_LAMBDA_TR',1,1.0_rk,'')
         END IF
      END IF

      CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR+DELTA_RR,STATEV(8),
     1             F_LOK+DELTA_F_LOK,R,FSTAR,HH,DFSTAR_DFEFF,
     2             EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,-1.0_rk)

      IF (PNEW==0) THEN
         IF (FSTAR .NE. FSTAR) THEN
            PNEW=1
            CALL STDB_ABQERR(1,'NAN',1,1.0_rk,'')
            CALL STDB_ABQERR(1,'F_LOK+DELTA_F_LOK',1,1.0_rk,'')
         END IF
      END IF
C     End of Test on NAN
C
C     Stop the simulation when crack length along the liagment over
C     expected value  
      CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR+DELTA_RR,STATEV(8),
     1             F_LOK+DELTA_F_LOK,R,FSTAR,HH,DFSTAR_DFEFF,
     2             EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)

C Step increment control    
      IF (PNEW==0) THEN
C----> Increase the step increment if previous increment converges fast
         PNEWDT      =1.25_rk
C     Update the solution-dependent state variables
         STATEV(1)   =EPSILON_MISES+DELTA_LAMBDA_PL
         STATEV(2)   =RR+DELTA_RR
C     Update KAPPA = STATEV(3)
         IF (STATEV(3)>TEMP+DTEMP) THEN
         ELSE
            STATEV(3) =TEMP+DTEMP
         END IF

         CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR+DELTA_RR,
     1              STATEV(8),F_LOK+DELTA_F_LOK,R,FSTAR,HH,DFSTAR_DFEFF,
     2               EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)

         TAU_MISES    =TAU_MISES_T-3.0_rk*G*DELTA_LAMBDA_PL
         TAU_HYD      =TAU_HYD_T-3.0_rk*K*DELTA_LAMBDA_PLV
C     Stress triaxiality
         IF (TAU_MISES > 0.0_rk)THEN
            T   =TAU_HYD/TAU_MISES
         ELSE
            T   =1.0E8
         END IF
C
         IF (LOKAL_NLOKAL==1) THEN
            IF (SIGN(1.0_rk,FLIESSEN)>0.0_rk ) THEN
               IF (FCUMSCHALTEN<2.5) THEN
                  STATEV(5)   = STATEV(5) + DELTA_LAMBDA_PLV
     1                                           + DELTA_NUKLEATION
                  IF (F_N<=0.0_rk) THEN
                     DELTA_D =0.0_rk
                   ELSE
                     DELTA_D =DELTA_NUKLEATION*(1.0_rk-F_LOK
     1                                       -DELTA_F_LOK)*3.0_rk/F_N
                  END IF
               ELSE
                   STATEV(5)   =STATEV(5)
                   DRPLDT      =0.0_rk
                   DRPLDE      =0.0_rk
                   DELTA_D     =0.0_rk
               END IF

               STATEV(4)   =DAMAGE+DELTA_D
                
            ELSE
               DRPLDE  =0.0_rk
               DRPLDT  =0.0_rk
               DDSDDT  =0.0_rk
            END IF
             
            STATEV(4)   =DAMAGEOUT
         ELSE
            IF (FCUMSCHALTEN<2.5) THEN
               STATEV(5) = STATEV(5) + DELTA_LAMBDA_PLV
c!               write(*,*) statev(5)
            ELSE
               STATEV(5) = STATEV(5)
            END IF
         END IF
C     Check and update variables for material total damage
         IF (F_LOK+DELTA_F_LOK>=F_C .AND. FCUMSCHALTEN<1.5) THEN
            FCUMSCHALTEN        =2.0_rk
            STATEV(8)           =F_LOK+DELTA_F_LOK
         ELSEIF (FSTAR>=0.98_rk/Q_1) THEN
            FCUMSCHALTEN        =4.0_rk
         END IF
C     Update the solution-dependent variables
         STATEV(6)   =FCUMSCHALTEN
         STATEV(7)   =DRPLDT*DEPS_VOL
         STATEV(9)   =FSTAR
         STATEV(10)  =F_LOK+DELTA_F_LOK
         STATEV(11)  =T
C     Tranfer stress and material tangent stiffness matrix to Abaqus
         STRESS_MATRIX(1:NTENS,1)       =STRESS
         DDSDDE = (JAC*DDSDDE+MATMUL(STRESS_MATRIX,
     1                         TRANSPOSE(DJAC_DEPS)))/JAC
         STRESS      =STRESS
         DRPLDE      =DRPLDE
         DRPLDT      =(STATEV(7)-1.0_rk)*JAC
         RPL         =(STATEV(5)-TEMP-DTEMP)*JAC
         DDSDDT      =DDSDDT
      ELSE
C----> Reduce the step increment if previous increment converges slow
         PNEWDT      =1.0_rk/2.0_rk
         DDSDDE      =CMATRIX
         DDSDDT      =0.0_rk
         DRPLDE      =0.0_rk
         DRPLDT      =0.0_rk
         RPL         =0.0_rk
C
         STRESS      =0.0_rk
      END IF
C End of step increment control
C
      RETURN
      END SUBROUTINE UMAT
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine to obtain result of cross product
C on 2 vectors
C
      SUBROUTINE VECTOR_PRODUCT (NDI,NTENS,VEKTOR1,VEKTOR2,MATRIX)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Input vectors
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN) ::VEKTOR1,VEKTOR2
C     Matrix output
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(OUT)::MATRIX
C     Local variables declaration
      INTEGER(KIND=intk)                           ::i,j
      INTEGER(KIND=intk),INTENT(IN)                ::NTENS,NDI
C-----------------------------------------------------------------------
C
      MATRIX=0.0_rk
C
      DO i=1,NTENS,1
            DO j=1,NTENS,1
                  MATRIX(i,j)=VEKTOR1(i)*VEKTOR2(j)
            END DO
      END DO
C
      RETURN
      END SUBROUTINE VECTOR_PRODUCT
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C Definition of strain hardening
C
      SUBROUTINE STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RRIN,F_CIN,FEFF,
     1                            R,FSTAR,HH,DFSTAR_DFEFF,EPS_V,DT,DRR)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration
C     Variables passed in for information
      INTEGER(KIND=intk),INTENT(IN)::NPROPS
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)::PROPS
      CHARACTER(80),INTENT(IN)::CMNAME
      REAL(KIND=rk),INTENT(IN)    ::RRIN,F_CIN,EPS_V
      REAL(KIND=rk),INTENT(IN)    ::FEFF
      REAL(KIND=rk),INTENT(IN)    ::DT,DRR
C     Outputvariables
      REAL(KIND=rk),INTENT(OUT)   ::R,FSTAR
      REAL(KIND=rk),INTENT(OUT)   ::HH,DFSTAR_DFEFF
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk)     ::RR,FEFF_TOT,FZW,A_FSTAR,B_FSTAR,F_END
C     Local variables for one parametric powerlaw
      REAL(KIND=rk)     ::SYIELD,SYIELD_0,RN,RE,S0,EQPLAS,TOL
      LOGICAL                 ::FOUND
C     Local variables for tabullar input
      INTEGER(KIND=intk)   ::NPROPSUHARD
      REAL(KIND=rk),DIMENSION(NPROPS-13)::PROPSUHARD
      REAL(KIND=rk),DIMENSION(2,(NPROPS-13)/2)::TABLE
      REAL(KIND=rk)::SYIEL0,EQPL1,EQPL0
      INTEGER::I,NENTRIES
C     Intrinsic function
      INTRINSIC EXP,LOG,SQRT,EPSILON,HUGE,ABS
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for choosing options of hardening curve
      REAL(KIND=rk)   ::YIELDCURVE
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Critical porosity
      F_C  = F_CIN
C     Check equivalent plastic strain
      IF (RRIN<0.0_rk) THEN
        RR  =0.0_rk
      ELSE
        RR  =RRIN
      END IF
C     Current equivalent plastic strain
      EQPLAS   = RR
C     Strain hardening input option
C---->Strain hardening law as one parametric power law
      IF (NPROPS == 13) THEN
         S0       = TAU_0
         RN       = N
         RE       = E/S0
         TOL      = 1E-6
C
         IF (EQPLAS > 0.0_rk) THEN
            SYIELD      = 1.0_rk
            FOUND       = .true.
            DO WHILE (FOUND)
               SYIELD   =  (SYIELD+RE*EQPLAS)**RN
               FOUND    = ((SYIELD-SYIELD_0) > TOL)
               SYIELD_0 =  SYIELD
            END DO
         ELSE
            EQPLAS = 0.0_rk
            SYIELD = 1.0_rk
         END IF
         R     =  S0*SYIELD
         HH    =  R*RN*RE/((1-RN)*SYIELD+RE*EQPLAS)
C---->Strain hardening in tabular form (Yield stress - Plastic strain)
      ELSEIF (NPROPS > 13) THEN
         NPROPSUHARD = NPROPS - 13
         PROPSUHARD  = PROPS(14:)
C
         IF (NPROPSUHARD==1) THEN
             R=PROPS(1)
             HH=0.0_rk
             RETURN
         END IF
C
         IF (MOD(NPROPSUHARD,2)==1) THEN
            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
            CALL STDB_ABQERR(-3,'Odd number of elastoplastic'//
     1                  ' properties specified. Last entry is ignored.',
     2                    1,1.0_rk,'')
         END IF
C
         NENTRIES = NPROPSUHARD/2
         TABLE=RESHAPE(PROPSUHARD,[2,NENTRIES])
C     Yield stress = Last value of table
         R=TABLE(1,NENTRIES)
         HH=0.0_rk
C     Search and interpolate, if existing
         IF(NENTRIES>1) THEN
            I=1
C     First equivalent plastic strain always set to zero, even if 
C     specified incorrectly
            EQPL1=0.0_rk
            FOUND=.FALSE.
            DO WHILE (.NOT.(FOUND).AND.(I<NENTRIES))
               I=I+1
               EQPL0=EQPL1
               EQPL1=TABLE(2,I)
               IF(EQPL1<=EQPL0) THEN
                  CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
                  CALL STDB_ABQERR(-3,'ERROR in UHARD: plastic '//
     1                   ' strains are not entered in ascending '//
     2                    'order!!!',1,1.0_rk,'')
               ENDIF
               FOUND=(EQPLAS<EQPL1)
            END DO
C
            IF (FOUND) THEN
               SYIEL0=TABLE(1,I-1)
               HH=(TABLE(1,I)-SYIEL0)/(EQPL1-EQPL0)
               R=SYIEL0+(EQPLAS-EQPL0)*HH
            END IF
         END IF
      END IF
C     Porosity at material total damage
      F_END       = 0.995_rk/Q_1
      FEFF_TOT    = 0.98_rk/Q_1
C     Check material state under damage condition     
      IF (FEFF<F_C .OR. KAPPA<EPSILON(0.0_rk)) THEN
         FSTAR           =FEFF
         DFSTAR_DFEFF    =1.0_rk
        IF (FSTAR>FEFF_TOT) THEN
            FZW          = FEFF_TOT

            A_FSTAR      = KAPPA/(F_END-FEFF_TOT)

            B_FSTAR      = FZW+1.0_rk/A_FSTAR*LOG(1.0_rk-FEFF_TOT/F_END)

            FSTAR        = F_END*(1.0_rk-EXP(-A_FSTAR*(FEFF-B_FSTAR)))

            DFSTAR_DFEFF = F_END*A_FSTAR*EXP(-A_FSTAR*(FEFF-B_FSTAR))
        END IF
C
      ELSE
         FSTAR           =F_C+KAPPA*(FEFF-F_C)
         DFSTAR_DFEFF    =KAPPA
         IF (FSTAR>FEFF_TOT) THEN
            FZW          = (FEFF_TOT-F_C)/KAPPA+F_C

            A_FSTAR      = KAPPA/(F_END-FEFF_TOT)

            B_FSTAR      = FZW+1.0_rk/A_FSTAR*LOG(1.0_rk-FEFF_TOT/F_END)

            FSTAR        = F_END*(1.0_rk-EXP(-A_FSTAR*(FEFF-B_FSTAR)))

            DFSTAR_DFEFF = F_END*A_FSTAR*EXP(-A_FSTAR*(FEFF-B_FSTAR))
         END IF
      END IF
C
      RETURN
      END SUBROUTINE STRAIN_HARDENING
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine to obtain matrix inversion of 2x2 matrix
      SUBROUTINE MATRIX_INVERSION_2_2 (MATRIX, INVERSEMATRIX)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Declare input matrix
      REAL(KIND=rk),DIMENSION(2,2),INTENT(IN)   ::MATRIX
C     Declare output matrix
      REAL(KIND=rk),DIMENSION(2,2),INTENT(OUT)  ::INVERSEMATRIX 
C     Local variables
      REAL(KIND=rk)                             ::J_DET
C-----------------------------------------------------------------------
C
      IF (SIZE(MATRIX) .EQ. 4) THEN
    
      ELSE
         CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
         CALL STDB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO WRONG ',
     1       'MATRIX INPUT. IT SHOULD BE 2X2 MATRIX',1,1.0_rk,'')
      END IF
C
      J_DET = MATRIX(1,1)*MATRIX(2,2)-MATRIX(1,2)*MATRIX(2,1)
C    
      INVERSEMATRIX(1,1) =  MATRIX(2,2)
      INVERSEMATRIX(1,2) = -MATRIX(1,2)
C    
      INVERSEMATRIX(2,1) = -MATRIX(2,1)
      INVERSEMATRIX(2,2) =  MATRIX(1,1)
C    
      INVERSEMATRIX      =  1.0_rk/J_DET*INVERSEMATRIX
C
      RETURN
      END SUBROUTINE MATRIX_INVERSION_2_2
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C Calculate the updated stress and material tangen stiffness matrix
C
      SUBROUTINE PLASTICITY_MODULE_NL(DELTA_E_EQ_GES,DTIME,NTENS,
     1             NPROPS,NSTATV,STARTDELTA, TAU_T,DEVIATORT,N_TR,
     2             CMATRIX,KRONECKER,IMATRIX,IDEV,NORMMATRIX,
     3             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,
     4             STRESS,DDSDDE,DDSDDT,DRPLDE,DRPLDT,DELTA_LAMBDA_PL,
     5             DELTA_LAMBDA_PLV,DELTA_RR,
     6             DELTA_F_EFF,DELTA_NUKLEATION,PNEW,CMNAME)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration
C     Variables passed in for information
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(IN) ::CMATRIX,
     1                                          IMATRIX,IDEV,NORMMATRIX
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN)       ::TAU_T,
     1                                          DEVIATORT,N_TR,KRONECKER
      INTEGER(KIND=intk),INTENT(IN)                      ::NTENS,
     1                                          NPROPS,NSTATV
      REAL(KIND=rk),DIMENSION(2),INTENT(IN)           ::STARTDELTA
      REAL(KIND=rk),INTENT(IN) ::I1T,J2T,TAU_MISES_T,TAU_HYD_T,
     1                                     DELTA_E_EQ_GES,DELTA_F_EFF
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)   ::PROPS
      REAL(KIND=rk),DIMENSION(NSTATV),INTENT(IN)   ::STATEV
      REAL(KIND=rk),INTENT(IN)                     ::DTIME
      CHARACTER(80),INTENT(IN)::CMNAME
C     Outputvariables
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(OUT) ::STRESS,DDSDDT
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(OUT)  ::DDSDDE
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(OUT)        ::DRPLDE
      REAL(KIND=rk),INTENT(OUT)  ::DRPLDT
      REAL(KIND=rk),INTENT(OUT)  ::DELTA_LAMBDA_PL,DELTA_RR,
     1                               DELTA_LAMBDA_PLV,DELTA_NUKLEATION
      INTEGER(KIND=intk),INTENT(INOUT)        ::PNEW
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk)     ::EPSILON_MISES,RR,FCUMSCHALTEN,FEFF
      REAL(KIND=rk)     ::R,FSTAR,HH,DFSTAR_DFEFF
      INTEGER(KIND=intk)   ::MAX_ITER,ZAEHLER
      REAL(KIND=rk)     ::TOL,ERROR
      REAL(KIND=rk),DIMENSION(2,2)        ::KMATRIX,KMATRIXINVERSE
      REAL(KIND=rk),DIMENSION(2)          ::RMATRIX
      REAL(KIND=rk),DIMENSION(2,NTENS)    ::BMATRIX
      REAL(KIND=rk)                 ::L1, L2
      REAL(KIND=rk),DIMENSION(2)    ::LMATRIX
      REAL(KIND=rk),DIMENSION(1,NTENS)          ::A1,A2
      REAL(KIND=rk),DIMENSION(2,NTENS)          ::AMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,NTENS)      ::PHIMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,1)  ::DEVIATORTMATRIX,
     1                                     KRONECKERMATRIX,N_TRMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,NTENS)::CMATRIX_TANGENT
      REAL(KIND=rk),DIMENSION(4)      ::NUKLEATION
C     Intrinsic function
      INTRINSIC MATMUL,TRANSPOSE
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for choosing options of hardening curve
      REAL(KIND=rk)   ::YIELDCURVE
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Caculation bulk modulus and shear modulus
      K           = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G           = E/(2.0_rk*(1.0_rk+NU))
C     Assign solution-dependent state variables to local variables
      EPSILON_MISES  =STATEV(1)
      RR             =STATEV(2)
      FCUMSCHALTEN   =STATEV(6)
      FEFF           =STATEV(10)
C     Check and assign initial value for Newton-Raphson-Loop
      IF (DELTA_LAMBDA_PL<=0.0_rk) THEN
            DELTA_LAMBDA_PL         =0.0_rk
      ELSE
            DELTA_LAMBDA_PL         =STARTDELTA(1)
      END IF
      DELTA_LAMBDA_PLV        =STARTDELTA(2)    
      DELTA_RR                =0.0_rk
C     Fixed parameter for Newton-Raphson-Loop
      TOL                     =1E-10_rk
      ERROR                   =2.0_rk*TOL
      MAX_ITER                =30
      ZAEHLER                 =0
      PNEW                    =0
C     Newton-Raphson-Loop for micro plastic strain
      DO WHILE (ERROR>=TOL .AND. PNEW==0)
C
        CALL DERIVATIVES_NL (DTIME,NTENS,NPROPS,NSTATV,DELTA_LAMBDA_PL,
     1             DELTA_LAMBDA_PLV,DEVIATORT,KRONECKER,NORMMATRIX,IDEV,
     2             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,0,KMATRIX,
     3          RMATRIX,BMATRIX,DELTA_RR,DELTA_F_EFF,NUKLEATION,,CMNAME)

        CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
        LMATRIX = -MATMUL(KMATRIXINVERSE,RMATRIX)
        L1 = LMATRIX(1)
        L2 = LMATRIX(2)

        IF (L1 .NE. L1) THEN
              L1    =DELTA_LAMBDA_PL*0.1
        END IF      

        IF (L2 .NE. L2) THEN
              L2    =DELTA_LAMBDA_PLV*0.01
        END IF      

        DELTA_LAMBDA_PL         =DELTA_LAMBDA_PL+L1
        DELTA_LAMBDA_PLV        =DELTA_LAMBDA_PLV+L2

        ERROR       =sqrt(RMATRIX(1)**2+RMATRIX(2)**2+L1**2+L2**2)

        IF (ZAEHLER>=MAX_ITER) THEN
            PNEW    =1
c!            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!            CALL STDB_ABQERR(1,'#Iteration for micro plastic strain'//
c!     1             ' and non-local variable exceed the limit of 30',
c!     2               1,1.0_rk,'')
        END IF
        
        ZAEHLER=ZAEHLER+1

      END DO 
C     End of Newton-Raphson-Loop
C
C     Stress update
      IF (PNEW==0) THEN
        IF (DELTA_LAMBDA_PL==0.0_rk .AND.  
     1                         DELTA_LAMBDA_PLV ==0.0_rk) THEN
c!            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!            CALL STDB_ABQERR(1,'No improvement of plastic multiplier',
c!     1                    1,1.0_rk,'')
        END IF
C
        CALL DERIVATIVES_NL (DTIME,NTENS,NPROPS,NSTATV,DELTA_LAMBDA_PL,
     1             DELTA_LAMBDA_PLV,DEVIATORT,KRONECKER,NORMMATRIX,IDEV,
     2             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,1,KMATRIX,
     3          RMATRIX,BMATRIX,DELTA_RR,DELTA_F_EFF,NUKLEATION,,CMNAME)
        
        CMATRIX_TANGENT   =MATMUL(NORMMATRIX,CMATRIX)
        STRESS    =TAU_T-MATMUL(CMATRIX_TANGENT,(DELTA_LAMBDA_PL*N_TR
     1                         +DELTA_LAMBDA_PLV*KRONECKER))

      ELSE
        STRESS    =TAU_T
      END IF
C     Material tangent stiffness matrix
      IF (PNEW==0) THEN
C     Nucleation damage
        DELTA_NUKLEATION   = F_n*NUKLEATION(1)
     1                         /(1.0_rk-FEFF-DELTA_F_EFF)/3.0_rk
C     
        CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
        AMATRIX = matmul(KMATRIXINVERSE,BMATRIX)
        A1(1,1:NTENS) = AMATRIX(1,1:NTENS)
        A2(1,1:NTENS) = AMATRIX(2,1:NTENS)

        DEVIATORTMATRIX(1:NTENS,1)    =DEVIATORT(1:NTENS)
        KRONECKERMATRIX(1:NTENS,1)    =KRONECKER(1:NTENS)
        N_TRMATRIX(1:NTENS,1)         =N_TR(1:NTENS)

        PHIMATRIX   =(IDEV-(3.0_rk/2.0_rk/TAU_MISES_T**2)
     1                  *MATMUL(DEVIATORTMATRIX,
     2                   MATMUL(TRANSPOSE(DEVIATORTMATRIX),NORMMATRIX)))
     3                  *3.0_rk/2.0_rk/TAU_MISES_T

C     Derivative of thermal source with respect to strain increment
        DRPLDE(1:NTENS) =A2(1,1:NTENS)+F_N/(1.0_rk-FEFF-DELTA_F_EFF)
     1                         /3.0_rk*NUKLEATION(2)*A1(1,1:NTENS)
C     Derivative of updated stress with respect to strain increment
        DDSDDE =IMATRIX-MATMUL(CMATRIX_TANGENT,(DELTA_LAMBDA_PL
     1                     *PHIMATRIX+MATMUL(N_TRMATRIX,A1)
     2                     +MATMUL(KRONECKERMATRIX,A2)))
C     
        CALL DERIVATIVES_NL (DTIME,NTENS,NPROPS,NSTATV,DELTA_LAMBDA_PL,
     1             DELTA_LAMBDA_PLV,DEVIATORT,KRONECKER,NORMMATRIX,IDEV,
     2             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,2,KMATRIX,
     3           RMATRIX,BMATRIX,DELTA_RR,DELTA_F_EFF,NUKLEATION,CMNAME)

        CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
        LMATRIX = -matmul(KMATRIXINVERSE,RMATRIX)
        L1 = LMATRIX(1)
        L2 = LMATRIX(2)
C     Derivative of thermal source with respect to temperature
        DRPLDT  =L2+F_N/(1.0_rk-FEFF-DELTA_F_EFF)/3.0_rk*NUKLEATION(2)
     1                   *L1+F_N*NUKLEATION(1)/3.0_rk
     2                   /(1.0_rk-FEFF-DELTA_F_EFF)**2
C     Derivative of updated stress with respect to temperature
        DDSDDT  =-MATMUL(CMATRIX_TANGENT,N_TR*L1+KRONECKER*L2)
      ELSE
         DDSDDE            =IMATRIX
         DDSDDT            =0.0_rk
         DELTA_NUKLEATION  =0.0_rk
      END IF
C
      RETURN
      END SUBROUTINE PLASTICITY_MODULE_NL

C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C
      SUBROUTINE DERIVATIVES_NL (DTIME,NTENS,NPROPS,NSTATV,
     1               DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,
     2               DEVIATORT,KRONECKER,NORMMATRIX,IDEV,PROPS,STATEV,
     3               I1T,J2T,TAU_MISES_T,TAU_HYD_T,FALL,
     4               KMATRIX,RMATRIX,BMATRIX,DELTA_RR,
     5               DELTA_F_EFF,NUKLEATION,CMNAME)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration for DERIVATIVES_NL Subroutine
C     Input variables
      CHARACTER(80),INTENT(IN)::CMNAME
      REAL(KIND=rk),INTENT(IN)   ::DTIME,DELTA_F_EFF
      INTEGER(KIND=intk),INTENT(IN) ::NTENS,NPROPS,NSTATV,FALL
      REAL(KIND=rk),INTENT(IN)   ::DELTA_LAMBDA_PL,I1T,J2T,
     1                         TAU_MISES_T,TAU_HYD_T,DELTA_LAMBDA_PLV
      REAL(KIND=rk),DIMENSION(NSTATV),INTENT(IN)::STATEV
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)::PROPS
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN) ::
     1                                           DEVIATORT,KRONECKER
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(IN)::
     1                                           NORMMATRIX,IDEV
C     Output variables
      REAL(KIND=rk),DIMENSION(2,2),INTENT(OUT)       ::KMATRIX
      REAL(KIND=rk),DIMENSION(2),INTENT(OUT)         ::RMATRIX
      REAL(KIND=rk),DIMENSION(2,NTENS),INTENT(OUT)   ::BMATRIX
      REAL(KIND=rk),INTENT(OUT)                      ::DELTA_RR
C     Nucleation fraction of void in the model
      REAL(KIND=rk),DIMENSION(4),INTENT(OUT)    ::NUKLEATION
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk) ::EPSILON_MISES,RR
      REAL(KIND=rk) ::R,FSTAR,HH,DFSTAR_DFEFF
      REAL(KIND=rk) ::TAU_MISES,TAU_HYD
      REAL(KIND=rk) ::DR1_DTAU_MISES,DTAU_MISES_DLAMBDA_PL,DR1_DR,
     1                         DR_DRR,DDELTA_R_DDELTA_LAMBDA_PL,
     5                         DR2_DTAU_MISES,DR2_DTAU_HYD,
     6                         DTAU_HYD_DDELTA_LAMBDA_PL
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::DTAU_MISES_DTAU_T
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::DTAU_HYD_DTAU_T
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::B1,B2
      REAL(KIND=rk),DIMENSION(1,NTENS)    ::B1TR,B2TR

      REAL(KIND=rk)                       ::FEFF

      REAL(KIND=rk)::COS_H,SIN_H,D2R1_DTAU_MISES2,
     1                         D2R1_DTAU_MISES_DR,D2R1_DTAU_HYD2,
     2                         D2R1_DTAU_HYD_DFSTAR,D2R1_DTAU_HYD_DR,
     3                         DR1_DTAU_HYD,DTAU_HYD_DDELTA_LAMBDA_PLV,
     4                         DR1_DFSTAR

      REAL(KIND=rk)  ::EPS_EQ,DEPS_EQ_DLAMBDA_PL,
     1                         DEPS_EQ_DLAMBDA_PLV,ZUSATZ,ZUSATZ_EPS_EQ,
     2                         ZUSATZ_LAMBDA_PL,ZUSATZ_LAMBDA_PLV,
     3                         ZUSATZ_TAU_MISES,ZUSATZ_TAU_HYD,
     4                         ZUSATZ_DELTA_FEFF
C
      REAL(KIND=rk)  ::TOL,ERROR
      INTEGER(KIND=intk)::ZAEHLER,MAXIMUM
C     Intrinsic function
      INTRINSIC MATMUL,ATAN,EPSILON,EXP,LOG,HUGE,COSH,SINH
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for choosing options of hardening curve
      REAL(KIND=rk)   ::YIELDCURVE
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Explicit number definition
      REAL(KIND=rk),PARAMETER::PI=3.141592653589793_rk
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Caculation bulk modulus and shear modulus
      K           = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G           = E/(2.0_rk*(1.0_rk+NU))
C     Assign solution-dependent state variables to local variables
      EPSILON_MISES  =STATEV(1)
      RR             =STATEV(2)
      FEFF           =STATEV(10)
C     Calculate equivalent Mises-stress and hydrostatic stress
      TAU_MISES    =TAU_MISES_T-3.0_rk*G*DELTA_LAMBDA_PL
      TAU_HYD      =TAU_HYD_T-3.0_rk*K*DELTA_LAMBDA_PLV
C     Assign values for variables used in Newton-Raphson-Loop
      TOL         =1E-10_rk
      ERROR       =2.0_rk*TOL 
      ZAEHLER     =0 
      MAXIMUM     =10 
      EPS_EQ      =DELTA_LAMBDA_PL 
C  
      DTAU_MISES_DLAMBDA_PL       =-3.0_rk*G 
      DTAU_HYD_DDELTA_LAMBDA_PLV  =-3.0_rk*K 
C                
      CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR,STATEV(8),
     1                   FEFF+DELTA_F_EFF,R,FSTAR,HH,DFSTAR_DFEFF,
     2                   EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,EPS_EQ) 
C     Newton-Raphson-Loop for hardening variable
      DO WHILE (ERROR>TOL .AND. ZAEHLER<=MAXIMUM)
                    
            DELTA_RR     =EPS_EQ
             
            CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR+DELTA_RR,
     1               STATEV(8),FEFF+DELTA_F_EFF,R,FSTAR,HH,DFSTAR_DFEFF,
     2                   EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)
                    
            ZUSATZ  =EPS_EQ*R-(TAU_MISES*DELTA_LAMBDA_PL
     1                   +3.0_rk*TAU_HYD*DELTA_LAMBDA_PLV)
     2                   /(1.0_rk-FEFF-DELTA_F_EFF) 
                    
            DR_DRR                      =HH 
            DDELTA_R_DDELTA_LAMBDA_PL   =1.0_rk 
                    
            ZUSATZ_EPS_EQ =R+EPS_EQ*DR_DRR
     1                                    *DDELTA_R_DDELTA_LAMBDA_PL 
                    
            EPS_EQ      =EPS_EQ-ZUSATZ/ZUSATZ_EPS_EQ 
                    
            ERROR       =abs(ZUSATZ)+abs(ZUSATZ/ZUSATZ_EPS_EQ) 
                    
            ZAEHLER     =ZAEHLER+1
            IF (ZAEHLER==MAXIMUM) THEN
c!               CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!               CALL STDB_ABQERR(1,'#Iteration for hardening'//
c!     1             ' variable exceeds the limit of 10',1,1.0_rk,'')
            END IF
      END DO
C     End of Newton-Raphson-Loop for hardening variable
C
C     Derivative of hardening variable
      DELTA_RR     =EPS_EQ 
      CALL STRAIN_HARDENING (PROPS,NPROPS,STATEV(8),RR+DELTA_RR,
     1               STATEV(8),FEFF+DELTA_F_EFF,R,FSTAR,HH,DFSTAR_DFEFF,
     2                   EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)     
                
      DR_DRR                      =HH 
      DDELTA_R_DDELTA_LAMBDA_PL   =1.0_rk 

C     Derivatives
      ZUSATZ_EPS_EQ   =R+EPS_EQ*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL 
                
      ZUSATZ_LAMBDA_PL    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                   *(TAU_MISES+DELTA_LAMBDA_PL
     2                   *DTAU_MISES_DLAMBDA_PL)
                
      ZUSATZ_LAMBDA_PLV    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                   *3.0_rk*(TAU_HYD+DELTA_LAMBDA_PLV
     2                   *DTAU_HYD_DDELTA_LAMBDA_PLV)
                            
      DEPS_EQ_DLAMBDA_PL   =-ZUSATZ_LAMBDA_PL/ZUSATZ_EPS_EQ 
                
      DEPS_EQ_DLAMBDA_PLV  =-ZUSATZ_LAMBDA_PLV/ZUSATZ_EPS_EQ 
                
      COS_H        =COSH(Q_2*3.0_rk/2.0_rk*TAU_HYD/R)
      SIN_H        =SINH(Q_2*3.0_rk/2.0_rk*TAU_HYD/R)

C     Derivatives of yield function
      DR1_DTAU_MISES         =2.0_rk*TAU_MISES/R**2
      D2R1_DTAU_MISES2       =2.0_rk/R**2
      D2R1_DTAU_MISES_DR     =-4.0_rk*TAU_MISES/R**3 
      D2R1_DTAU_HYD2         =9.0_rk/2.0_rk*Q_1*Q_2**2*FSTAR/R**2*COS_H 
      D2R1_DTAU_HYD_DFSTAR   =3.0_rk*Q_1*Q_2/R*SIN_H 
      D2R1_DTAU_HYD_DR       =-3.0_rk*Q_1*Q_2*FSTAR/R**2*SIN_H
     1                               -9.0_rk*Q_2**2*Q_1*FSTAR*TAU_HYD
     2                               /2.0_rk/R**3*COS_H 
      DR1_DTAU_HYD           =3.0_rk*Q_1*Q_2*FSTAR/R*SIN_H 
      DR1_DR                 =-2.0_rk*TAU_MISES**2/R**3-3.0_rk*Q_1*Q_2
     1                               *FSTAR*TAU_HYD/R**2*SIN_H 
      DR1_DFSTAR             =2.0_rk*Q_1*COS_H-2.0_rk*Q_1**2*FSTAR
C     Assemble K-Matrix
      KMATRIX       =0.0_rk

      KMATRIX(1,1)  =DR1_DTAU_MISES*DTAU_MISES_DLAMBDA_PL+DR1_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PL
                
      KMATRIX(1,2)  =DR1_DTAU_HYD*DTAU_HYD_DDELTA_LAMBDA_PLV
                
      KMATRIX(1,2)  =KMATRIX(1,2)+DR1_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PLV 
                
      KMATRIX(2,1)  =DELTA_LAMBDA_PLV*(D2R1_DTAU_MISES2
     1                   *DTAU_MISES_DLAMBDA_PL+D2R1_DTAU_MISES_DR
     2                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     3                   *DEPS_EQ_DLAMBDA_PL) 
                
      KMATRIX(2,1)  =KMATRIX(2,1)-DR1_DTAU_HYD*1.0_rk/3.0_rk
                
      KMATRIX(2,1)  =KMATRIX(2,1)-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *D2R1_DTAU_HYD_DR*DR_DRR
     2                   *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PL                  
                
      KMATRIX(2,2)  =DR1_DTAU_MISES-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *(D2R1_DTAU_HYD2*DTAU_HYD_DDELTA_LAMBDA_PLV) 
                
      KMATRIX(2,2)  =KMATRIX(2,2)+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PLV 
                
      KMATRIX(2,2)  =KMATRIX(2,2)-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *D2R1_DTAU_HYD_DR*DR_DRR
     2                   *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PLV 
      
C     Option for this subroutine between Newton-Raphson-Loop for micro
C     plastic strain and material tangent stiffness matrix
      SELECT CASE (FALL)
C
         CASE (0)
C     Residue output for Newton-Raphson-Loop
           RMATRIX      =0.0_rk
           RMATRIX(1)   =TAU_MISES**2/R**2+2.0_rk*Q_1*FSTAR*COS_H
     1                         -1.0_rk-Q_1**2*FSTAR**2
           RMATRIX(2)   =DELTA_LAMBDA_PLV*DR1_DTAU_MISES-DELTA_LAMBDA_PL
     1                         *1.0_rk/3.0_rk*DR1_DTAU_HYD
           BMATRIX      =0.0_rk
C
         CASE (1)
C     Output to calculate material tangent stiffness matrix
            DTAU_MISES_DTAU_T(1:NTENS,1)  =3.0_rk/2.0_rk/TAU_MISES_T
     1                                     *MATMUL(NORMMATRIX,DEVIATORT)
            DTAU_HYD_DTAU_T(1:NTENS,1)    =1.0_rk/3.0_rk*KRONECKER

            ZUSATZ_TAU_MISES    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                               *DELTA_LAMBDA_PL
 
            ZUSATZ_TAU_MISES    =-ZUSATZ_TAU_MISES/ZUSATZ_EPS_EQ
 
            ZUSATZ_TAU_HYD      =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                               *3.0_rk*DELTA_LAMBDA_PLV
 
            ZUSATZ_TAU_HYD      =-ZUSATZ_TAU_HYD/ZUSATZ_EPS_EQ
C   
            BMATRIX     =0.0_rk
 
            B1    =DR1_DTAU_MISES*DTAU_MISES_DTAU_T
     1                   +DR1_DTAU_HYD*DTAU_HYD_DTAU_T
            B1    =B1+DR1_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_MISES*DTAU_MISES_DTAU_T 
            B1    =B1+DR1_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_HYD*DTAU_HYD_DTAU_T 
            
            B1TR                =-TRANSPOSE(B1)
            BMATRIX(1,1:NTENS)  =B1TR(1,1:NTENS)                      
                    
            B2    =DELTA_LAMBDA_PLV*D2R1_DTAU_MISES2*DTAU_MISES_DTAU_T 
            B2    =B2+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_TAU_MISES
     1                   *DTAU_MISES_DTAU_T 
            B2    =B2+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_TAU_HYD
     1                   *DTAU_HYD_DTAU_T 
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD2
     1                   *DTAU_HYD_DTAU_T 
     
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_MISES*DTAU_MISES_DTAU_T 
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *ZUSATZ_TAU_HYD*DTAU_HYD_DTAU_T 
                        
            B2TR                 =-TRANSPOSE(B2)
            BMATRIX(2,1:NTENS)   =B2TR(1,1:NTENS)
                  
            RMATRIX      =0.0_rk
C
         CASE (2)    
           ZUSATZ_DELTA_FEFF  =-(TAU_MISES*DELTA_LAMBDA_PL+3.0_rk
     1                               *TAU_HYD*DELTA_LAMBDA_PLV)
     1                               /(1.0_rk-FEFF-DELTA_F_EFF)**2 
           ZUSATZ_DELTA_FEFF  =-ZUSATZ_DELTA_FEFF/ZUSATZ_EPS_EQ 
     
           RMATRIX     =0.0_rk 
     
           RMATRIX(1)  =DR1_DFSTAR*DFSTAR_DFEFF+DR1_DR*DR_DRR
     1                  *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_DELTA_FEFF
     
           RMATRIX(2)  =-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                  *D2R1_DTAU_HYD_DFSTAR*DFSTAR_DFEFF
     2                  +DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR*DR_DRR
     3                  *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_DELTA_FEFF
     4                  -DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD_DR
     5                  *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     6                  *ZUSATZ_DELTA_FEFF
     
           BMATRIX             =0.0_rk

      END SELECT
C     Nucleation damage after Chu-Needleman 
      NUKLEATION(1) =1.0_rk/(S_n*SQRT(2.0_rk*PI))*EXP(-1.0_rk/2.0_rk
     1                  *((EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n)**2)
     2                  *DELTA_LAMBDA_PL
      NUKLEATION(2) =1.0_rk/(S_n*SQRT(2.0_rk*PI))*EXP(-1.0_rk/2.0_rk
     1                  *((EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n)**2)
     2                  -DELTA_LAMBDA_PL*1.0_rk/(S_n**2*SQRT(2.0_rk*PI))
     3                  *EXP(-1.0_rk/2.0_rk*((EPSILON_MISES
     4                  +DELTA_LAMBDA_PL-E_n)/S_n)**2)
     5                  *(EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n
      NUKLEATION(3)    =0.0_rk
      NUKLEATION(4)    =0.0_rk
C
      RETURN
      END SUBROUTINE DERIVATIVES_NL
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C
c!      SUBROUTINE PLASTIZITAETS_MODUL(DELTA_E_EQ_GES,DTIME,NTENS,NPROPS,
c!     1                   NSTATV,STARTDELTA, TAU_T,DEVIATORT,N_TR,CMATRIX,
c!     2                   KRONECKER,IMATRIX,IDEV,NORMMATRIX,PROPS,STATEV,
c!     3                   I1T,J2T,J3T,COS3THETA,TAU_MISES_T,TAU_HYD_T,
c!     4                   STRESS,DDSDDE,DELTA_LAMBDA_PL,DELTA_LAMBDA_TR,
c!     5                   DELTA_LAMBDA_PLV,DELTA_RR,DELTA_F_EFF,DAMAGEOUT,
c!     6                   PNEW,F_SB)
      SUBROUTINE PLASTICITY_MODULE(DELTA_E_EQ_GES,DTIME,NTENS,NPROPS,
     1                   NSTATV,STARTDELTA,TAU_T,DEVIATORT,N_TR,CMATRIX,
     2                   KRONECKER,IMATRIX,IDEV,NORMMATRIX,PROPS,STATEV,
     3                   I1T,J2T,TAU_MISES_T,TAU_HYD_T,STRESS,DDSDDE,
     4                   DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,DELTA_RR,
     5                   DELTA_F_EFF,DAMAGEOUT,DELTA_NUKLEATION,PNEW)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration
C     Variables passed in for information
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(IN) ::CMATRIX,
     1                                          IMATRIX,IDEV,NORMMATRIX
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN)       ::TAU_T,
     1                                          DEVIATORT,N_TR,KRONECKER
      INTEGER(KIND=intk),INTENT(IN)                      ::NTENS,
     1                                          NPROPS,NSTATV
      REAL(KIND=rk),DIMENSION(2),INTENT(IN)           ::STARTDELTA
      REAL(KIND=rk),INTENT(IN) ::I1T,J2T,TAU_MISES_T,TAU_HYD_T,
     1                                     DELTA_E_EQ_GES
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)   ::PROPS
      REAL(KIND=rk),DIMENSION(NSTATV),INTENT(IN)   ::STATEV
      REAL(KIND=rk),INTENT(IN)                     ::DTIME
C     Outputvariables
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(OUT)        ::STRESS
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(OUT)  ::DDSDDE
      REAL(KIND=rk),INTENT(OUT)  ::DELTA_LAMBDA_PL,DELTA_RR,
     1               DELTA_LAMBDA_PLV,DAMAGEOUT,DELTA_NUKLEATION
      INTEGER(KIND=intk),INTENT(INOUT)        ::PNEW,DELTA_F_EFF
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk)     ::EPSILON_MISES,RR,FCUMSCHALTEN,FEFF
      REAL(KIND=rk)     ::R,FSTAR,HH,DFSTAR_DFEFF
      INTEGER(KIND=intk)   ::MAX_ITER,ZAEHLER
      REAL(KIND=rk)     ::TOL,ERROR
      REAL(KIND=rk),DIMENSION(2,2)        ::KMATRIX,KMATRIXINVERSE
      REAL(KIND=rk),DIMENSION(2)          ::RMATRIX
      REAL(KIND=rk),DIMENSION(2,NTENS)    ::BMATRIX
      REAL(KIND=rk)                 ::L1, L2
      REAL(KIND=rk),DIMENSION(2)    ::LMATRIX
      REAL(KIND=rk),DIMENSION(1,NTENS)          ::A1,A2
      REAL(KIND=rk),DIMENSION(2,NTENS)          ::AMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,NTENS)      ::PHIMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,1)  ::DEVIATORTMATRIX,
     1                                     KRONECKERMATRIX,N_TRMATRIX
      REAL(KIND=rk),DIMENSION(NTENS,NTENS)::CMATRIX_TANGENT
      REAL(KIND=rk),DIMENSION(4)      ::NUKLEATION
C     Intrinsic function
      INTRINSIC MATMUL,TRANSPOSE
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for choosing options of hardening curve
      REAL(KIND=rk)   ::YIELDCURVE
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Caculation bulk modulus and shear modulus
      K           = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G           = E/(2.0_rk*(1.0_rk+NU))
C     Assign solution-dependent state variables to local variables
      EPSILON_MISES  =STATEV(1)
      RR             =STATEV(2)
      FCUMSCHALTEN   =STATEV(6)
      FEFF           =STATEV(10)
C     Check and assign initial value for Newton-Raphson-Loop
      IF (DELTA_LAMBDA_PL<=0.0_rk) THEN
            DELTA_LAMBDA_PL         =0.0_rk
      ELSE
            DELTA_LAMBDA_PL         =STARTDELTA(1)
      END IF
      DELTA_LAMBDA_PLV        =STARTDELTA(2)    
      DELTA_RR                =0.0_rk
C     Fixed parameter for Newton-Raphson-Loop
      TOL                     =1E-10_rk
      ERROR                   =2.0_rk*TOL
      MAX_ITER                =30
      ZAEHLER                 =0
      PNEW                    =0
C     Newton-Raphson-Loop for micro equivalent plastic strain
C     and micro volumetric plastic strain
      DO WHILE (ERROR>=TOL .AND. PNEW==0)
C
         CALL DERIVATIVES (DTIME,NTENS,NPROPS,NSTATV,DELTA_LAMBDA_PL,
     1             DELTA_LAMBDA_PLV,DEVIATORT,KRONECKER,NORMMATRIX,IDEV,
     2             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,0,KMATRIX,
     3             RMATRIX,BMATRIX,DELTA_RR,DELTA_F_EFF,NUKLEATION)

        CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
        LMATRIX = -MATMUL(KMATRIXINVERSE,RMATRIX)
        L1 = LMATRIX(1)
        L2 = LMATRIX(2)

        IF (L1 .NE. L1) THEN
              L1    =DELTA_LAMBDA_PL*0.1
        END IF      

        IF (L2 .NE. L2) THEN
              L2    =DELTA_LAMBDA_PLV*0.01
        END IF      

        DELTA_LAMBDA_PL         =DELTA_LAMBDA_PL+L1
        DELTA_LAMBDA_PLV        =DELTA_LAMBDA_PLV+L2

        ERROR       =sqrt(RMATRIX(1)**2+RMATRIX(2)**2+L1**2+L2**2)

        IF (ZAEHLER>=MAX_ITER) THEN
            PNEW    =1
c!            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!            CALL STDB_ABQERR(1,'#Iteration for micro plastic strain'//
c!     1             ' and non-local variable exceed the limit of 30',
c!     2               1,1.0_rk,'')
        END IF
C
        ZAEHLER=ZAEHLER+1
      END DO 
C     End of Newton-Raphson-Loop
C
C     Stress update
      IF (PNEW==0) THEN
         IF (DELTA_LAMBDA_PL==0.0_rk .AND.  
     1                         DELTA_LAMBDA_PLV ==0.0_rk) THEN
c!            CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!            CALL STDB_ABQERR(1,'No improvement of plastic multiplier',
c!     1                    1,1.0_rk,'')
         END IF
C
         CALL DERIVATIVES (DTIME,NTENS,NPROPS,NSTATV,DELTA_LAMBDA_PL,
     1             DELTA_LAMBDA_PLV,DEVIATORT,KRONECKER,NORMMATRIX,IDEV,
     2             PROPS,STATEV,I1T,J2T,TAU_MISES_T,TAU_HYD_T,1,KMATRIX,
     3             RMATRIX,BMATRIX,DELTA_RR,DELTA_F_EFF,NUKLEATION)
             
         CMATRIX_TANGENT   =MATMUL(NORMMATRIX,CMATRIX)
         STRESS    =TAU_T-MATMUL(CMATRIX_TANGENT,(DELTA_LAMBDA_PL*N_TR
     1                         +DELTA_LAMBDA_PLV*KRONECKER))

      ELSE
         STRESS    =TAU_T
      END IF
C     Material tangent stiffness matrix
      IF (PNEW==0) THEN
C     Nucleation damage
         DELTA_NUKLEATION   = F_N*NUKLEATION(1)
     1                         /(1.0_rk-FEFF-DELTA_F_EFF)/3.0_rk
C     
         CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
         AMATRIX = matmul(KMATRIXINVERSE,BMATRIX)
         A1(1,1:NTENS) = AMATRIX(1,1:NTENS)
         A2(1,1:NTENS) = AMATRIX(2,1:NTENS)
         
         DEVIATORTMATRIX(1:NTENS,1)    =DEVIATORT(1:NTENS)
         KRONECKERMATRIX(1:NTENS,1)    =KRONECKER(1:NTENS)
         N_TRMATRIX(1:NTENS,1)         =N_TR(1:NTENS)
         
         PHIMATRIX   =(IDEV-(3.0_rk/2.0_rk/TAU_MISES_T**2)
     1                  *MATMUL(DEVIATORTMATRIX,
     2                   MATMUL(TRANSPOSE(DEVIATORTMATRIX),NORMMATRIX)))
     3                  *3.0_rk/2.0_rk/TAU_MISES_T

C     Derivative of updated stress with respect to strain increment
        DDSDDE =IMATRIX-MATMUL(CMATRIX_TANGENT,(DELTA_LAMBDA_PL
     1                     *PHIMATRIX+MATMUL(N_TRMATRIX,A1)
     2                     +MATMUL(KRONECKERMATRIX,A2)))
      ELSE
         DDSDDE            =IMATRIX
         DELTA_NUKLEATION  =0.0_rk
      END IF
C
      DAMAGEOUT = 0.0_rk
      RETURN
      END SUBROUTINE PLASTICITY_MODULE
C***********************************************************************
C=======================================================================
C***********************************************************************
      SUBROUTINE DERIVATIVES (DTIME,NTENS,NPROPS,NSTATV,
     1               DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,
     2               DEVIATORT,KRONECKER,NORMMATRIX,IDEV,PROPS,STATEV,
     3               I1T,J2T,TAU_MISES_T,TAU_HYD_T,FALL,
     4               KMATRIX,RMATRIX,BMATRIX,DELTA_RR,
     5               DELTA_F_EFF,NUKLEATION)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration for DERIVATIVES_NL Subroutine
C     Input variables
      REAL(KIND=rk),INTENT(IN)   ::DTIME
      INTEGER(KIND=intk),INTENT(IN) ::NTENS,NPROPS,NSTATV,FALL
      REAL(KIND=rk),INTENT(IN)   ::DELTA_LAMBDA_PL,I1T,J2T,
     1                         TAU_MISES_T,TAU_HYD_T,DELTA_LAMBDA_PLV
      REAL(KIND=rk),DIMENSION(NSTATV),INTENT(IN)::STATEV
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)::PROPS
      REAL(KIND=rk),DIMENSION(NTENS),INTENT(IN) ::
     1                                           DEVIATORT,KRONECKER
      REAL(KIND=rk),DIMENSION(NTENS,NTENS),INTENT(IN)::
     1                                           NORMMATRIX,IDEV
C     Output variables
      REAL(KIND=rk),DIMENSION(2,2),INTENT(OUT)       ::KMATRIX
      REAL(KIND=rk),DIMENSION(2),INTENT(OUT)         ::RMATRIX
      REAL(KIND=rk),DIMENSION(2,NTENS),INTENT(OUT)   ::BMATRIX
      REAL(KIND=rk),INTENT(OUT)                      ::DELTA_RR
C     Nucleation fraction of void in the model
      REAL(KIND=rk),DIMENSION(4),INTENT(OUT)    ::NUKLEATION
      REAL(KIND=rk),INTENT(INOUT)               ::DELTA_F_EFF
C-----------------------------------------------------------------------
C     Local variables
      CHARACTER(80) ::CMNAME
      REAL(KIND=rk) ::EPSILON_MISES,RR,DAMAGE,FCUMSCHALTEN
      REAL(KIND=rk) ::R,FSTAR,HH,DFSTAR_DFEFF
      REAL(KIND=rk) ::TAU_MISES,TAU_HYD
      REAL(KIND=rk) ::DR1_DTAU_MISES,DTAU_MISES_DLAMBDA_PL,DR1_DR,
     1                         DR_DRR,DDELTA_R_DDELTA_LAMBDA_PL,
     2                         DR2_DTAU_MISES,DR2_DTAU_HYD,
     3                         DTAU_HYD_DDELTA_LAMBDA_PL
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::DTAU_MISES_DTAU_T
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::DTAU_HYD_DTAU_T
      REAL(KIND=rk),DIMENSION(NTENS,1)    ::B1,B2
      REAL(KIND=rk),DIMENSION(1,NTENS)    ::B1TR,B2TR

      REAL(KIND=rk)                       ::FEFF

      REAL(KIND=rk)::COS_H,SIN_H,D2R1_DTAU_MISES2,
     1                         D2R1_DTAU_MISES_DR,D2R1_DTAU_HYD2,
     2                         D2R1_DTAU_HYD_DFSTAR,D2R1_DTAU_HYD_DR,
     3                         DR1_DTAU_HYD,DTAU_HYD_DDELTA_LAMBDA_PLV,
     4                         DR1_DFSTAR

      REAL(KIND=rk)  ::EPS_EQ,DEPS_EQ_DLAMBDA_PL,
     1                         DEPS_EQ_DLAMBDA_PLV,ZUSATZ,ZUSATZ_EPS_EQ,
     2                         ZUSATZ_LAMBDA_PL,ZUSATZ_LAMBDA_PLV,
     3                         ZUSATZ_TAU_MISES,ZUSATZ_TAU_HYD,
     4                         ZUSATZ_DELTA_FEFF
C     Local variables for void growth and void nucleation
      REAL(KIND=rk) ::NUC_FACTOR,DELTA_F_EFF_NUC
      REAL(KIND=rk) ::DDELTA_F_EFF_NUC_DDELTA_LAMBDA_PL
      REAL(KIND=rk) ::DDELTA_FEFF_DDELTA_LAMBDA_PL
      REAL(KIND=rk) ::DDELTA_FEFF_DDELTA_LAMBDA_PLV
C
      REAL(KIND=rk)  ::TOL,ERROR
      INTEGER(KIND=intk)::ZAEHLER,MAXIMUM
C     Intrinsic function
      INTRINSIC ABS,MATMUL,COSH,SINH,TRANSPOSE,SQRT,EXP
C-----------------------------------------------------------------------
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for choosing options of hardening curve
      REAL(KIND=rk)   ::YIELDCURVE
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_n,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Explicit number definition
      REAL(KIND=rk),PARAMETER::PI=3.141592653589793_rk
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Assign solution-dependent state variables to local variables
      EPSILON_MISES  =STATEV(1)
      RR             =STATEV(2)
      DAMAGE         =STATEV(4)
      FCUMSCHALTEN   =STATEV(6)
      FEFF           =STATEV(10)
C     Calculate damage evolution (Nucleation and Growth)
      NUC_FACTOR       = F_n/(S_n*SQRT(2.0_rk*PI))
     1                 *EXP(-1.0_rk/2.0_rk*((EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n)**2)
      DELTA_F_EFF_NUC  = NUC_FACTOR*DELTA_LAMBDA_PL
      
      DDELTA_F_EFF_NUC_DDELTA_LAMBDA_PL =F_n/(S_n*SQRT(2.0_rk*PI))*EXP(-1.0_rk/2.0_rk
     1                  *((EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n)**2)
     2                  -DELTA_LAMBDA_PL*F_n/(S_n**2*SQRT(2.0_rk*PI))
     3                  *EXP(-1.0_rk/2.0_rk*((EPSILON_MISES
     4                  +DELTA_LAMBDA_PL-E_n)/S_n)**2)
     5                  *(EPSILON_MISES+DELTA_LAMBDA_PL-E_n)/S_n/S_n
     
      DELTA_F_EFF  =(3.0_rk*(1.0_rk-FEFF)*DELTA_LAMBDA_PLV 
     1                +DELTA_F_EFF_NUC)/(1.0_rk+3.0_rk*DELTA_LAMBDA_PLV)
     
      DDELTA_FEFF_DDELTA_LAMBDA_PL  = DDELTA_F_EFF_NUC_DDELTA_LAMBDA_PL
     1                  /(1.0_rk+3.0_rk*DELTA_LAMBDA_PLV)
      DDELTA_FEFF_DDELTA_LAMBDA_PLV = 3.0_rk*(1.0_rk-FEFF
     1             -DELTA_F_EFF_NUC)/(1.0_rk+3.0_rk*DELTA_LAMBDA_PLV)**2
C-----------------------------------------------------------------------
C     Caculation bulk modulus and shear modulus
      K           = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G           = E/(2.0_rk*(1.0_rk+NU))
C     Calculate equivalent Mises-stress and hydrostatic stress
      TAU_MISES    =TAU_MISES_T-3.0_rk*G*DELTA_LAMBDA_PL
      TAU_HYD      =TAU_HYD_T-3.0_rk*K*DELTA_LAMBDA_PLV
C     Derivatives of Mises-stress and hydrostatic stress
      DTAU_MISES_DLAMBDA_PL       =-3.0_rk*G 
      DTAU_HYD_DDELTA_LAMBDA_PLV  =-3.0_rk*K 
C     Assign values for variables used in Newton-Raphson-Loop
      TOL         =1E-10_rk
      ERROR       =2.0_rk*TOL 
      ZAEHLER     =0 
      MAXIMUM     =10 
C     Instead of Zero, assign a better initial value for NR-Loop variable
      EPS_EQ      =DELTA_LAMBDA_PL              
C     Newton-Raphson-Loop to calculate hardening equivalent plastic strain
      DO WHILE (ERROR>TOL .AND. ZAEHLER<=MAXIMUM)
C-------->>>Call subroutine to calculate current yield stress and hard modulus
            DELTA_RR     =EPS_EQ
C
            CALL STRAIN_HARDENING (PROPS,NPROPS,CMNAME,RR+DELTA_RR,
     1               STATEV(8),FEFF+DELTA_F_EFF,R,FSTAR,HH,DFSTAR_DFEFF,
     2                   EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)
C-------->>>Calculate the function f3 in the paper of Andreas Seupel
            ZUSATZ  =EPS_EQ*R-(TAU_MISES*DELTA_LAMBDA_PL
     1                   +3.0_rk*TAU_HYD*DELTA_LAMBDA_PLV)
     2                   /(1.0_rk-FEFF-DELTA_F_EFF) 
C-------->>>Intermediate derivative of yield stress w.r.t hardening variable
            DR_DRR        =HH 
C-------->>>Derivative of function f3 w.r.t hardening variable EPS_EQ
            ZUSATZ_EPS_EQ =R+EPS_EQ*DR_DRR
C-------->>>Hardening variable
            EPS_EQ      =EPS_EQ-ZUSATZ/ZUSATZ_EPS_EQ 
                    
            ERROR       =abs(ZUSATZ)+abs(ZUSATZ/ZUSATZ_EPS_EQ) 
                    
            ZAEHLER     =ZAEHLER+1
            IF (ZAEHLER==MAXIMUM) THEN
c!               CALL STDB_ABQERR(1,'=== Message from GTN UMAT by TU'//
c!     1                     ' Bergakademie Freiberg ===',1,1.0_rk,'')
c!               CALL STDB_ABQERR(1,'#Iteration for hardening'//
c!     1             ' variable exceeds the limit of 10',1,1.0_rk,'')
            END IF
      END DO
C     End of Newton-Raphson-Loop for hardening variable
C
C     Call subroutine to calculate current yield stress and hard modulus
C     based on the converged value of NR-Loop
C
      DELTA_RR     =EPS_EQ 
      CALL STRAIN_HARDENING (PROPS,NPROPS,STATEV(8),RR+DELTA_RR,
     1               STATEV(8),FEFF+DELTA_F_EFF,R,FSTAR,HH,DFSTAR_DFEFF,
     2                   EPSILON_MISES+DELTA_LAMBDA_PL,DTIME,DELTA_RR)     
                
      DR_DRR                      =HH 
      DDELTA_R_DDELTA_LAMBDA_PL   =1.0_rk 

      ZUSATZ_EPS_EQ   =R+EPS_EQ*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL 
                
      ZUSATZ_LAMBDA_PL    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1      *(TAU_MISES+DELTA_LAMBDA_PL*DTAU_MISES_DLAMBDA_PL)
     2      -TAU_MISES*DELTA_LAMBDA_PL
     3      *DDELTA_FEFF_DDELTA_LAMBDA_PL/(1.0_rk-FEFF-DELTA_F_EFF)**2
                
      ZUSATZ_LAMBDA_PLV    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1     *3.0_rk*(TAU_HYD+DELTA_LAMBDA_PLV*DTAU_HYD_DDELTA_LAMBDA_PLV)
     2     -3.0_rk*TAU_HYD*DELTA_LAMBDA_PLV
     3     *DDELTA_FEFF_DDELTA_LAMBDA_PLV/(1.0_rk-FEFF-DELTA_F_EFF)**2
                            
      DEPS_EQ_DLAMBDA_PL   =-ZUSATZ_LAMBDA_PL/ZUSATZ_EPS_EQ 
                
      DEPS_EQ_DLAMBDA_PLV  =-ZUSATZ_LAMBDA_PLV/ZUSATZ_EPS_EQ 
                
      COS_H        =COSH(Q_2*3.0_rk/2.0_rk*TAU_HYD/R)
      SIN_H        =SINH(Q_2*3.0_rk/2.0_rk*TAU_HYD/R)

C     Derivatives of yield function
      DR1_DTAU_MISES         =2.0_rk*TAU_MISES/R**2
      D2R1_DTAU_MISES2       =2.0_rk/R**2
      D2R1_DTAU_MISES_DR     =-4.0_rk*TAU_MISES/R**3 
      D2R1_DTAU_HYD2         =9.0_rk/2.0_rk*Q_1*Q_2**2*FSTAR/R**2*COS_H 
      D2R1_DTAU_HYD_DFSTAR   =3.0_rk*Q_1*Q_2/R*SIN_H 
      D2R1_DTAU_HYD_DR       =-3.0_rk*Q_1*Q_2*FSTAR/R**2*SIN_H
     1                               -9.0_rk*Q_2**2*Q_1*FSTAR*TAU_HYD
     2                               /2.0_rk/R**3*COS_H 
      DR1_DTAU_HYD           =3.0_rk*Q_1*Q_2*FSTAR/R*SIN_H 
      DR1_DR                 =-2.0_rk*TAU_MISES**2/R**3-3.0_rk*Q_1*Q_2
     1                               *FSTAR*TAU_HYD/R**2*SIN_H 
      DR1_DFSTAR             =2.0_rk*Q_1*COS_H-2.0_rk*Q_1**2*FSTAR
C     Assemble K-Matrix
      KMATRIX       =0.0_rk

      KMATRIX(1,1)  =DR1_DTAU_MISES*DTAU_MISES_DLAMBDA_PL+DR1_DR
     1      *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PL
     2      +DR1_DFSTAR*DFSTAR_DFEFF*DDELTA_FEFF_DDELTA_LAMBDA_PL 
                
      KMATRIX(1,2)  =DR1_DTAU_HYD*DTAU_HYD_DDELTA_LAMBDA_PLV
     1      +DR1_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PLV
     2      +DR1_DFSTAR*DFSTAR_DFEFF*DDELTA_FEFF_DDELTA_LAMBDA_PLV 
                
      KMATRIX(2,1)  =DELTA_LAMBDA_PLV*(D2R1_DTAU_MISES2
     1      *DTAU_MISES_DLAMBDA_PL+D2R1_DTAU_MISES_DR*DR_DRR
     2      *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PL) 
                
      KMATRIX(2,1)  =KMATRIX(2,1)-1.0_rk/3.0_rk*DR1_DTAU_HYD
                
      KMATRIX(2,1)  =KMATRIX(2,1)-1.0_rk/3.0_rk*DELTA_LAMBDA_PL
     1      *D2R1_DTAU_HYD_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2      *DEPS_EQ_DLAMBDA_PL
     
      KMATRIX(2,1)  =KMATRIX(2,1)-1.0_rk/3.0_rk*DELTA_LAMBDA_PL
     1      *D2R1_DTAU_HYD_DFSTAR*DFSTAR_DFEFF
     2      *DDELTA_FEFF_DDELTA_LAMBDA_PL                
                
      KMATRIX(2,2)  =DR1_DTAU_MISES-1.0_rk/3.0_rk*DELTA_LAMBDA_PL
     1      *(D2R1_DTAU_HYD2*DTAU_HYD_DDELTA_LAMBDA_PLV) 
                
      KMATRIX(2,2)  =KMATRIX(2,2)+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR
     1      *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PLV 

      KMATRIX(2,2)  =KMATRIX(2,2)-1.0_rk/3.0_rk*DELTA_LAMBDA_PL
     1                   *D2R1_DTAU_HYD_DR*DR_DRR
     2                   *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PLV
     
      KMATRIX(2,2)  =KMATRIX(2,2)-1.0_rk/3.0_rk*DELTA_LAMBDA_PL
     1      *D2R1_DTAU_HYD_DFSTAR*DFSTAR_DFEFF
     2      *DDELTA_FEFF_DDELTA_LAMBDA_PLV
      
C     Option for this subroutine between Newton-Raphson-Loop for micro
C     plastic strain and material tangent stiffness matrix
      SELECT CASE (FALL)
C
         CASE (0)
C     Residue output for Newton-Raphson-Loop
            RMATRIX      =0.0_rk
            RMATRIX(1)   =TAU_MISES**2/R**2+2.0_rk*Q_1*FSTAR*COS_H
     1                         -1.0_rk-Q_1**2*FSTAR**2
            RMATRIX(2)   =DELTA_LAMBDA_PLV*DR1_DTAU_MISES-DELTA_LAMBDA_PL
     1                         *1.0_rk/3.0_rk*DR1_DTAU_HYD
            BMATRIX      =0.0_rk
C
         CASE (1)
C     Output to calculate material tangent stiffness matrix
            DTAU_MISES_DTAU_T(1:NTENS,1)  =3.0_rk/2.0_rk/TAU_MISES_T
     1                                     *MATMUL(NORMMATRIX,DEVIATORT)
            DTAU_HYD_DTAU_T(1:NTENS,1)    =1.0_rk/3.0_rk*KRONECKER

            ZUSATZ_TAU_MISES    =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                               *DELTA_LAMBDA_PL
 
            ZUSATZ_TAU_MISES    =-ZUSATZ_TAU_MISES/ZUSATZ_EPS_EQ
 
            ZUSATZ_TAU_HYD      =-1.0_rk/(1.0_rk-FEFF-DELTA_F_EFF)
     1                               *3.0_rk*DELTA_LAMBDA_PLV
 
            ZUSATZ_TAU_HYD      =-ZUSATZ_TAU_HYD/ZUSATZ_EPS_EQ
            
            ZUSATZ_DELTA_FEFF    =-(TAU_MISES*DELTA_LAMBDA_PL
     1             +3.0_rk*TAU_HYD*DELTA_LAMBDA_PLV)
     2             /(1.0_rk-FEFF-DELTA_F_EFF)**2

            ZUSATZ_DELTA_FEFF    =-ZUSATZ_DELTA_FEFF/ZUSATZ_EPS_EQ
C   
            BMATRIX     =0.0_rk
 
            B1    =DR1_DTAU_MISES*DTAU_MISES_DTAU_T
     1                   +DR1_DTAU_HYD*DTAU_HYD_DTAU_T
            B1    =B1+DR1_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_MISES*DTAU_MISES_DTAU_T 
            B1    =B1+DR1_DR*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_HYD*DTAU_HYD_DTAU_T 
            
            B1TR                =-TRANSPOSE(B1)
            BMATRIX(1,1:NTENS)  =B1TR(1,1:NTENS)                      
                    
            B2    =DELTA_LAMBDA_PLV*D2R1_DTAU_MISES2*DTAU_MISES_DTAU_T 
            B2    =B2+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_TAU_MISES
     1                   *DTAU_MISES_DTAU_T 
            B2    =B2+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL*ZUSATZ_TAU_HYD
     1                   *DTAU_HYD_DTAU_T 
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD2
     1                   *DTAU_HYD_DTAU_T 
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     1                   *ZUSATZ_TAU_MISES*DTAU_MISES_DTAU_T 
            B2    =B2-DELTA_LAMBDA_PL*1.0_rk/3.0_rk*D2R1_DTAU_HYD_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *ZUSATZ_TAU_HYD*DTAU_HYD_DTAU_T 
                        
            B2TR                 =-TRANSPOSE(B2)
            BMATRIX(2,1:NTENS)   =B2TR(1,1:NTENS)
                  
            RMATRIX      =0.0_rk
C
      END SELECT
C
      RETURN
      END SUBROUTINE DERIVATIVES
