C=======================================================================
C TU Bergakademie Freiberg
C Faculty of Mechanical, Process and Energy Engineering
C Institute for Mechanics and Fluid Dynamics (IMFD)
************************************************************************
C Implementation of non-local GTN-Model for ductile damage in ABAQUS
C Version 1.0
C Date 17 May 2022
C Author: Andreas Seupel, Geralf Hütter, Rinh Pham, Omar El Khatib,
C         Bjoern Kiefer
************************************************************************
C Corresponding author:
C     andreas.seupel@imfd.tu-freiberg.de (A.Seupel)
C     geralf.huetter@imfd.tu-freiberg.de (G.Hütter)
C     dinh-rinh.pham1@imfd.tu-freiberg.de (R. Pham)
C Reference:
C G.Hütter, T.Linse, U.Mühlich and M.Kuna: Simulation of Ductile Crack 
C           Initiation and Propagation by means of a Non-local GTN-model 
C           under Small-Scale Yielding, International Journal of Solids 
C           and Structures, 50 (2013), 662--671.
C A.Seupel, G.Hütter, and M.~Kuna: On the identification and uniqueness 
C           of constitutive parameters for a non-local GTN-model, 
C           Engineering Fracture Mechanics, 229 (2020), 106817.
C R.D. Pham, O. El Khatib, A. Seupel, G. Hütter and B. Kiefer: 
C           Non-iterative determination of the parameters for the Gurson
C           model from two standard tests, DVM-Bericht 254(2022), 43-52
C O.El Khatib, G.Hütter, D.R.Pham, A.~Seupel, M.~Kuna and B.~Kiefer: 
C           A non-iterative parameter calibration procedure for the 
C           GTN model based on standardized experiments,"in preparation".
C***********************************************************************
C=======================================================================
C***********************************************************************
      MODULE Abaqus_Interface
            INCLUDE 'VABA_PARAM.INC'
            PRIVATE
            INTEGER,PARAMETER,PUBLIC::rk=KIND(r),intk=KIND(i)
      END MODULE
C***********************************************************************
C=======================================================================
C***********************************************************************
      subroutine vumat(
C Read only -
     1  nblock, ndir, nshr, nstatev, nfieldv, nprops, lanneal,
     2  stepTime, totalTime, dt, cmname, coordMp, charLength,
     3  props, density, strainInc, relSpinInc,
     4  tempOld, stretchOld, defgradOld, fieldOld,
     3  stressOld, stateOld, enerInternOld, enerInelasOld,
     6  tempNew, stretchNew, defgradNew, fieldNew,
C Write only -
     5  stressNew, stateNew, enerInternNew, enerInelasNew )
C Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C Explicit definition of variables
      IMPLICIT NONE
C     Standard variables for Umat subroutine in ABAQUS
      INTEGER(KIND=intk),INTENT(IN) :: nblock, ndir, nshr, nstatev, nfieldv, nprops, lanneal
      REAL(KIND=rk),INTENT(IN) :: stepTime, totalTime, dt
      CHARACTER(80)::CMNAME
      REAL(KIND=rk),INTENT(IN) :: coordMp(nblock,3),charLength(nblock)
      REAL(KIND=rk),DIMENSION(nprops),INTENT(IN) :: props
      REAL(KIND=rk),DIMENSION(nblock),INTENT(IN) ::density, tempOld, tempNew, enerInternOld, enerInelasOld
      REAL(KIND=rk),DIMENSION(nblock,ndir+nshr),INTENT(IN) :: strainInc, stretchOld, stressOld, stretchNew
      REAL(KIND=rk),DIMENSION(nblock,nshr),INTENT(IN) :: relSpinInc
      REAL(KIND=rk),DIMENSION(nblock,ndir+2*nshr),INTENT(IN) :: defgradOld, defgradNew
      REAL(KIND=rk),DIMENSION(nblock,nfieldv),INTENT(IN) :: fieldOld, fieldNew
      REAL(KIND=rk),DIMENSION(nblock,nstatev),INTENT(IN) :: stateOld

      REAL(KIND=rk),DIMENSION(nblock,ndir+nshr),INTENT(OUT) :: stressNew
      REAL(KIND=rk),DIMENSION(nblock,nstatev),INTENT(OUT)   :: stateNew
      REAL(KIND=rk),DIMENSION(nblock):: enerInternNew, enerInelasNew
C     Local variables declaration
      INTEGER(KIND=intk) :: kblock
      REAL(KIND=rk)      :: e,xnu,twomu,alamda,thremu
      REAL(KIND=rk)      :: jacobian
      REAL(KIND=rk)      :: s11,s22,s33,s12,s13,s23,smean,vmisestr
      REAL(KIND=rk)      :: ntr11,ntr22,ntr33,ntr12,ntr13,ntr23,trace_ntr
      REAL(KIND=rk)      :: trace,sigdif,facyld,dpeeqmicro,deqpshardening
      REAL(KIND=rk)      :: yieldNew, factor
      REAL(KIND=rk)      :: stressPower
      REAL(KIND=rk)      :: plasticWorkInc
C
      REAL(KIND=rk)      ::vmisesNew,shydNew
C     Local variables declaration for VUHARD
      INTEGER(KIND=intk) :: nvalue
      REAL(KIND=rk)      ::yieldOld, hard
      REAL(KIND=rk)      ::peeqOldhardening
C     Local variable w.r.t state variable
      REAL(KIND=rk)      :: peeqOldmicro, peeqNewmicro 
      REAL(KIND=rk)      :: hardening,nl_vol_strain,damage_nuc,local_vol_strain
      REAL(KIND=rk)      :: damage_stt,f_cnu,fstarOld,feffOld,striaxiltiy
      REAL(KIND=rk)      :: feffNew
C     Local variable in yield function
      REAL(KIND=rk)      :: yieldtr
      REAL(KIND=rk)      :: q1,q2
      REAL(KIND=rk)      :: shydtr,fstar,cos_h
C      REAL(KIND=rk),DIMENSION(2,(nprops-13)/2)      :: table
C     Local variables declaration for STRAIN_HARDENING
      REAL(KIND=rk)      ::eqplasin,syield,hardmodul
C     Local variables declaration for DAMAGE_EVO & void nucleation
      REAL(KIND=rk)      :: feff, feff_tot, dfstar_dfeff
      REAL(KIND=rk)      :: delta_feff, deps_vol
      REAL(KIND=rk)      :: delta_peeq_vol_local
      REAL(KIND=rk)      :: nucleation_factor, delta_nucleation
C     Local variables declaration for PLASTICITY_MODULE_NL
      REAL(KIND=rk) :: DELTA_E_EQ_GES,DTIME_INK
      INTEGER(KIND=intk)  :: nblock_cur,NTENS,NSTATV
      REAL(KIND=rk),DIMENSION(2) ::STARTDELTA
      REAL(KIND=rk),DIMENSION(ndir+nshr) ::TAU_T,DEVIATORT,N_TR,
     1                     KRONECKER
      REAL(KIND=rk),DIMENSION(ndir+nshr,ndir+nshr) ::CMATRIX,
     1                     IMATRIX,IDEV,NORMMATRIX,EPS
c!      REAL(KIND=rk),DIMENSION(nstatev)::STATEV  
      REAL(KIND=rk),DIMENSION(nblock,nstatev):: STATEV 
      REAL(KIND=rk) ::I1T,J2T,TAU_MISES_T,TAU_HYD_T,DELTA_F_EFF      
C     Output variable from subroutine plasticity_module_nl
      REAL(KIND=rk),DIMENSION(ndir+nshr)  ::STRESS,DDSDDT
      REAL(KIND=rk),DIMENSION(ndir+nshr,ndir+nshr)::DDSDDE
      REAL(KIND=rk),DIMENSION(ndir+nshr)  ::DRPLDE
      REAL(KIND=rk)  ::DRPLDT
C     Explicit parameter declaration
      REAL(KIND=rk),PARAMETER ::zero = 0.0_rk, one = 1.0_rk, 
     1  two = 2.0_rk, three = 3.0_rk,
     2  third = one/three, half = 0.5_rk, twoThirds = two/three,
     3  op5 = 1.5_rk
      REAL(KIND=rk),PARAMETER ::PI=3.141592653589793_rk
C-----------------------------------------------------------------------
C     Declaration of variables which is used for Newton-Raphson Loop
      REAL(KIND=rk),PARAMETER ::TOL=1E-10_rk
      REAL(KIND=rk)      ::ERROR
      INTEGER(KIND=intk),PARAMETER ::MAX_ITER=30_intk
      INTEGER(KIND=intk) ::NUM_ITER
      INTEGER(KIND=intk) ::PNEW
      REAL(KIND=rk)                 ::L1, L2
      REAL(KIND=rk),DIMENSION(2)    ::LMATRIX,RMATRIX
      REAL(KIND=rk),DIMENSION(2,2)  ::KMATRIX,KMATRIXINVERSE
      REAL(KIND=rk)  ::DELTA_LAMBDA_PL,DELTA_RR,DELTA_LAMBDA_PLV,
     1                        DELTA_F_LOK,DELTA_NUKLEATION
      REAL(KIND=rk),DIMENSION(4)    ::NUKLEATION
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_N,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Variables for material properties
C     Assign material properties and model parameters
C     Young's Modulus
      e               =PROPS(1)
C     Poisson's ratio
      xnu              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_n             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------

      e = props(1)
      xnu = props(2)
      twomu = e / ( one + xnu )
      alamda = xnu * twomu / ( one - two * xnu )
      thremu = op5 * twomu
C
      if ( stepTime .eq. zero ) then
         do kblock = 1, nblock
         trace = strainInc(kblock,1) + strainInc(kblock,2) + strainInc(kblock,3)
         stressNew(kblock,1) = stressOld(kblock,1)
     1      + twomu * strainInc(kblock,1) + alamda * trace
         stressNew(kblock,2) = stressOld(kblock,2)
     1     + twomu * strainInc(kblock,2) + alamda * trace
         stressNew(kblock,3) = stressOld(kblock,3)
     1     + twomu * strainInc(kblock,3) + alamda * trace
         stressNew(kblock,4)=stressOld(kblock,4) + twomu * strainInc(kblock,4)
            if ( nshr .gt. 1 ) then
            stressNew(kblock,5)=stressOld(kblock,5) + twomu * strainInc(kblock,5)
            stressNew(kblock,6)=stressOld(kblock,6) + twomu * strainInc(kblock,6)
            end if
         end do
      else
         do kblock = 1, nblock
C     Assign old state variables to local variables
            peeqOldmicro      =stateOld(kblock,1) !Equivalent plastic strain in Micro scale
            peeqOldhardening  =stateOld(kblock,2) !Equivalent plastic strain for strain hardening in Micro scale    
            nl_vol_strain     =stateOld(kblock,3) !Non-local driving force of damage, small kappa in equation 11
            damage_nuc        =stateOld(kblock,4)
            local_vol_strain  =stateOld(kblock,5)
            damage_stt        =stateOld(kblock,6)
            f_cnu             =stateOld(kblock,8)
            fstarOld          =stateOld(kblock,9)
            feffOld           =stateOld(kblock,10)
            striaxiltiy       =stateOld(kblock,11) !Stress triaxiality, not used for calculating in code
C     Calulate jacobian value
            if ( nshr .gt. 1 ) then
               jacobian = defgradNew(kblock,1)*(defgradNew(kblock,2)*defgradNew(kblock,3)
     1                            -defgradNew(kblock,8)*defgradNew(kblock,5))
               jacobian = jacobian - defgradNew(kblock,4)*(defgradNew(kblock,7)*defgradNew(kblock,3)
     1                            -defgradNew(kblock,6)*defgradNew(kblock,5))
               jacobian = jacobian + defgradNew(kblock,9)*(defgradNew(kblock,7)*defgradNew(kblock,8)
     1                            -defgradNew(kblock,6)*defgradNew(kblock,2))
            else
               jacobian = defgradNew(kblock,1)*defgradNew(kblock,2)*defgradNew(kblock,3)
               jacobian = jacobian - defgradNew(kblock,5)*defgradNew(kblock,4)*defgradNew(kblock,3)
            end if
C     Calculate trial stress
            trace = strainInc(kblock,1) + strainInc(kblock,2) + strainInc(kblock,3)
            s11 = stressOld(kblock,1) + twomu * strainInc(kblock,1) + alamda*trace
            s22 = stressOld(kblock,2) + twomu * strainInc(kblock,2) + alamda*trace
            s33 = stressOld(kblock,3) + twomu * strainInc(kblock,3) + alamda*trace
            s12 = stressOld(kblock,4) + twomu * strainInc(kblock,4)
            if ( nshr .gt. 1 ) then
               s13 = stressOld(kblock,5) + twomu * strainInc(kblock,5)
               s23 = stressOld(kblock,6) + twomu * strainInc(kblock,6)
            end if
C     Calculate hydrostatic stress of trial stress
            smean = 1.0_rk/3.0_rk * ( s11 + s22 + s33 )
            s11 = s11 - smean
            s22 = s22 - smean
            s33 = s33 - smean
C     Von-mises trial stress (equivalent trial stress)
            if ( nshr .eq. 1 ) then
               vmisestr=sqrt(1.5_rk*(s11*s11+s22*s22+s33*s33+2.0_rk*s12*s12))
            else
               vmisestr=sqrt( 1.5_rk * (s11*s11 + s22*s22 + s33 * s33 +
     1            2.0_rk*(s12 * s12 + s13 * s13 + s23 * s23) ))
            end if
C     Calculate trial flow stress
            ntr11 = 1.5_rk*s11/vmisestr
            ntr22 = 1.5_rk*s22/vmisestr
            ntr33 = 1.5_rk*s33/vmisestr
            ntr12 = 1.5_rk*s12/vmisestr
            ntr13 = 1.5_rk*s13/vmisestr
            ntr23 = 1.5_rk*s23/vmisestr
            trace_ntr = ntr11 + ntr22 + ntr33
C     Calculate the effective porosity
            if (nl_vol_strain .gt. tempNew(kblock)) then
               stateNew(kblock,3) = stateOld(kblock,3)
               feffNew     = feffOld
               delta_feff  =0.0_rk
            else
               feffNew     = 1.0_rk-(1.0_rk-F_0)*EXP(-3.0_rk*tempNew(kblock))
               delta_feff  = feffNew - feffOld
            end if
C     Call subroutine to calculate fstar value for trial step
            call damage_evo(nprops,props,f_cnu,feffNew,fstar,dfstar_dfeff)
C     Call subroutine to calculate yield stress for trial step
            if (nprops .eq. 13) then
               call hardening_curve(nprops,props,peeqOldhardening,yieldOld,hard)
            elseif (nprops .gt. 13) then
               nvalue = (nprops-13)/2
               call vuhard(yieldOld, hard, peeqOldhardening, props(14), nvalue) 
            else
               call XPLB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO'//
     1       ' WRONG INPUT BY KEYWORD *USER MATERIAL',1,1.0_rk,'')
            end if
C
C     Change names of local variables to the one used in yield function
            q1 = Q_1
            q2 = Q_2
            shydtr = smean
            cos_h  = COSH(1.5_rk*q2*shydtr/yieldOld)
C     Trial yield function value
            yieldtr= vmisestr**2/yieldOld**2 + 2.0_rk*q1*fstar*cos_h-(1+(q1*fstar)**2)
C     Check if loading in elastic or elastoplastic regime
            if (sign(1.0_rk,yieldtr) <= 0.0_rk) then
C-----------------------------------------------------------------------  
C           ELASTIC REGIME 
C
C Update the stress
C
               stressNew(kblock,1) = s11 + smean
               stressNew(kblock,2) = s22 + smean
               stressNew(kblock,3) = s33 + smean
               stressNew(kblock,4) = s12
               if ( nshr .gt. 1 ) then
                  stressNew(kblock,5) = s13
                  stressNew(kblock,6) = s23
               end if
C
C Update the internal state variable evolution
C
               delta_lambda_pl   =0.0_rk
               delta_lambda_plv  =0.0_rk
               delta_nucleation  =0.0_rk
               delta_rr          =0.0_rk
C
C Update the state variables in Elasic regime
C This step must be defined, even if the old and new values are the same
C If not defined, it delivers WRONG RESULT
C
               stateNew(kblock,1) = stateOld(kblock,1)
               stateNew(kblock,2) = stateOld(kblock,2)
            else
C-----------------------------------------------------------------------  
C           ELASTOPLASTIC REGIME
C-----------------------------------------------------------------------   
C     Section for GTN plasticity
C
C     Newton-Raphson iteration to calculate DELTA_LAMBDA_PL und DELTA_LAMBDA_PLV
C
C     Parameter for Newton-Raphson iteration
               ERROR                   =2.0_rk*TOL
               NUM_ITER                =0
               PNEW                    =0
C     Assign initial value from the beginning of Newton-Raphson Loop
               DELTA_LAMBDA_PL   =0.0_rk
               DELTA_LAMBDA_PLV  =0.0_rk
               DELTA_RR          =0.0_rk
C     Newton-Raphson-Loop for micro plastic strain (equivalent and volumetric)
               DO WHILE (ERROR>=TOL .AND. PNEW==0)
C         
                  nblock_cur = kblock      
                  CALL nonlocalderivative (nblock,nblock_cur,nstatev,stateOld,
     1                           nprops,props,feffNew,fstar,vmisestr,shydtr,
     2                           DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,DELTA_RR,
     3                           KMATRIX,RMATRIX)
         
                  CALL MATRIX_INVERSION_2_2(KMATRIX,KMATRIXINVERSE)
                  LMATRIX = -MATMUL(KMATRIXINVERSE,RMATRIX)
                  L1 = LMATRIX(1)
                  L2 = LMATRIX(2)
                  IF (L1 .NE. L1) THEN
                     write (*,*) 'L1 is NAN'
                     L1    =DELTA_LAMBDA_PL*0.1
                  END IF      
         
                  IF (L2 .NE. L2) THEN
                     write (*,*) 'L2 is NAN'
                     L2    =DELTA_LAMBDA_PLV*0.01
                  END IF      
         
                  DELTA_LAMBDA_PL         =DELTA_LAMBDA_PL+L1
                  DELTA_LAMBDA_PLV        =DELTA_LAMBDA_PLV+L2
         
                  ERROR =sqrt(RMATRIX(1)**2+RMATRIX(2)**2+L1**2+L2**2)
         
                  IF (NUM_ITER>=MAX_ITER) THEN
                     PNEW    =1
C                     call XPLB_ABQERR(1,'#Iteration for micro plastic'//
C     1               ' strain and non-local variable exceed the' //
C     2               ' limit of 30',1,1.0_rk,CMNAME)
                  END IF
                  NUM_ITER= NUM_ITER+1
               END DO 
C     End of Newton-Raphson-Loop
C
C Update the stress
C
               stressNew(kblock,1) = s11 + smean - DELTA_LAMBDA_PL*twomu*ntr11 
     1             - DELTA_LAMBDA_PLV*(twomu + 3.0_rk*alamda)
               stressNew(kblock,2) = s22 + smean - DELTA_LAMBDA_PL*twomu*ntr22
     1             - DELTA_LAMBDA_PLV*(twomu + 3.0_rk*alamda)
               stressNew(kblock,3) = s33 + smean - DELTA_LAMBDA_PL*twomu*ntr33
     1             - DELTA_LAMBDA_PLV*(twomu + 3.0_rk*alamda)
               stressNew(kblock,4) = s12 - DELTA_LAMBDA_PL*twomu*ntr12
               if ( nshr .gt. 1 ) then
                  stressNew(kblock,5) = s13 - DELTA_LAMBDA_PL*twomu*ntr13
                  stressNew(kblock,6) = s23 - DELTA_LAMBDA_PL*twomu*ntr23
               end if
C
C Update the state variables
C
               dpeeqmicro = DELTA_LAMBDA_PL
               deqpshardening = DELTA_RR
               peeqNewmicro   = peeqOldmicro + dpeeqmicro
               stateNew(kblock,1) = peeqNewmicro
               stateNew(kblock,2) = stateOld(kblock,2) + deqpshardening 
            end if
C
C Update the state variables
C
C It's already defined in the above section, where the new porosity is calculated
            if (nl_vol_strain .gt. tempNew(kblock)) then
               stateNew(kblock,3) = stateOld(kblock,3)
            else
               stateNew(kblock,3) = tempNew(kblock)
            end if
C
            if (damage_stt .lt. 2.5_rk) then
C----------->>>Growth of void fraction
               delta_peeq_vol_local   = DELTA_LAMBDA_PLV
C----------->>>Nucleation of void fraction under plastic strain controll
               nucleation_factor  = F_n/(S_n*SQRT(2.0_rk*PI))
     1                 *EXP(-1.0_rk/2.0_rk*((peeqNewmicro-E_n)/S_n)**2)
               delta_nucleation   = nucleation_factor*dpeeqmicro
     1                         /(1.0_rk-feffNew)/3.0_rk
               stateNew(kblock,5) = stateOld(kblock,5) + delta_peeq_vol_local + delta_nucleation
            else 
               stateNew(kblock,5) = stateOld(kblock,5)
            end if
C
            feff_tot = 0.98_rk/q1
C This material model can be switched to Von miseses plasticity. To do so,
C one need to assign q1=0.0_rk, or f_0=0.0_rk
C This IF-Condition to ensure the coding still running with q1=0.0_rk
            if ( feff_tot < huge(feff_tot) ) then
C----------->>> Updated state variables in case of Gurson model
               if (feffNew .ge. F_C .and. damage_stt .lt. 1.5_rk) then
                  damage_stt = 2.0_rk
                  stateNew(kblock,6) = damage_stt
                  stateNew(kblock,8) = feffNew
               elseif (fstar .ge. feff_tot) then
                  damage_stt = 4.0_rk
                  stateNew(kblock,6) = damage_stt
                  stateNew(kblock,8) = stateOld(kblock,8)
               end if
               stateNew(kblock,6) = 0.0_rk
               stateNew(kblock,8) = stateOld(kblock,8)
            else
C----------->>> Updated state variables in case of von Mises Plasticity
               stateNew(kblock,6) = 0.0_rk
               stateNew(kblock,8) = stateOld(kblock,8)
            end if
C
            stateNew(kblock,9) = fstar
            stateNew(kblock,10) = feffNew
C
            K           = E/(3.0_rk*(1.0_rk-2.0_rk*xnu))
            G           = E/(2.0_rk*(1.0_rk+xnu))
            vmisesNew   = vmisestr - 3.0_rk*G*DELTA_LAMBDA_PL
            shydNew     = shydtr - 3.0_rk*K*DELTA_LAMBDA_PLV
            if (vmisesNew > 0.0_rk) then
               stateNew(kblock,11) = vmisesNew/shydNew
            else
               stateNew(kblock,11) = 0.0_rk
            end if
C-------->>> This state variable will be called in HETVAL, large deformation
            stateNew(kblock,7) = jacobian
C 
         end do
      end if
C
      return
      end
C***********************************************************************
C=======================================================================
C***********************************************************************
      subroutine vuhard(syield, hard, eqplas, table, nvalue)
C Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C Explicit definition of variables
      IMPLICIT NONE
C     Dummy argument variables declaration for VUHARD
      INTEGER(KIND=intk) :: nvalue
      REAL(KIND=rk)      ::syield, hard, eqplas
      REAL(KIND=rk),DIMENSION(2,nvalue) :: table
C     Local variables declaration for VUHARD
      INTEGER(KIND=intk) :: k1
      REAL(KIND=rk)      :: eqpl1,eqpl0,deqpl
      REAL(KIND=rk)      :: syiel1,syiel0,dsyiel
C-----------------------------------------------------------------------
C
C Set yield stress to last value of table, hardening to zero
C
      syield=table(1, nvalue)
      hard=0.0_rk
C
C if more than one entry, search table
C
      if(nvalue.gt.1) then
         do k1=1, nvalue-1
            eqpl1=table(2,k1+1)
            if(eqplas.lt.eqpl1) then
               eqpl0=table(2, k1)
C
C yield stress and hardening
C
               deqpl=eqpl1-eqpl0
               syiel0=table(1, k1)
               syiel1=table(1, k1+1)
               dsyiel=syiel1-syiel0
               hard=dsyiel/deqpl
               syield=syiel0+(eqplas-eqpl0)*hard
               goto 10
            endif
         end do
   10 continue
      endif
C
      return
      end
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C Definition of strain hardening
      SUBROUTINE hardening_curve (NPROPS,PROPS,EQPLASIN,SYIELD,HARDMODUL)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration
C     Variables passed in for information
      INTEGER(KIND=intk),INTENT(IN)::NPROPS
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)::PROPS
      REAL(KIND=rk),INTENT(IN)::EQPLASIN
C     Outputvariables
      REAL(KIND=rk),INTENT(OUT)   ::SYIELD,HARDMODUL
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk)     :: EQPLAS
C     Local variables for one parametric powerlaw
      REAL(KIND=rk)     :: S0,RN,RE,SYIELD_R,SYIELD_R0
      LOGICAL           :: FOUND
C     Explicit parameter definition
      REAL(KIND=rk),PARAMETER::TOL=0.000001_rk
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_N,E_n,S
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_N             =PROPS(9)
C     Fit paramter of 
      S               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Assign material property to local variable
      S0       = TAU_0
      RN       = N
      RE       = E/S0
C     Check equivalent plastic strain
      IF (EQPLASIN<0.0_rk) THEN
         EQPLAS = 0.0_rk
      ELSE
         EQPLAS = EQPLASIN
      END IF
C     Calculate new yield stress and hardening modulus
      IF (EQPLAS > 0.0_rk) THEN
         SYIELD_R0   = 0.0_rk
         SYIELD_R    = 1.0_rk
         FOUND       = .true.
         DO WHILE (FOUND)
            SYIELD_R   =  (SYIELD_R+RE*EQPLAS)**RN
            FOUND    = ((SYIELD_R-SYIELD_R0) > TOL)
            SYIELD_R0 =  SYIELD_R
         END DO
      ELSE
         EQPLAS   = 0.0_rk
         SYIELD_R = 1.0_rk
      END IF
      SYIELD     =  S0*SYIELD_R
      HARDMODUL  =  SYIELD*RN*RE/((1-RN)*SYIELD_R+RE*EQPLAS)
C
      RETURN
      END SUBROUTINE hardening_curve
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard subroutine calculating damage evolution (porosity)
      SUBROUTINE damage_evo (NPROPS,PROPS,F_CIN,FEFF,FSTAR,DFSTAR_DFEFF)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Use module, which defines a object with the material parameters
C      USE PROP_MODULE
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration
C     Variables passed in for information
      INTEGER(KIND=intk),INTENT(IN)::NPROPS
      REAL(KIND=rk),DIMENSION(NPROPS),INTENT(IN)::PROPS
      REAL(KIND=rk),INTENT(IN)    ::F_CIN,FEFF
C     Outputvariables
      REAL(KIND=rk),INTENT(OUT)   ::FSTAR,DFSTAR_DFEFF
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk)     ::F_C_NUM
      REAL(KIND=rk)     ::F_END,FEFF_TOT
      REAL(KIND=rk)     ::FZW,A_FSTAR,B_FSTAR
C-----------------------------------------------------------------------
C     Declaration of TYPE, which will contains material PROPS
C      TYPE(PROPERTIES):: MATPARA
C-----------------------------------------------------------------------
C     Intrinsic function
      INTRINSIC EPSILON,LOG,EXP
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variable for material porosity
      REAL(KIND=rk)   ::F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
C-----------------------------------------------------------------------
C
C     Critical porosity (for numerical reason, it's not equal to F_C in
C     the .inp file)
      IF (F_CIN .gt. F_C) THEN
         F_C_NUM  = F_CIN
      ELSE
         F_C_NUM  = F_C
      END IF
C  IF-Condition for no damage evolution (switched to Von Mises Plasticity)
      IF (Q_1 == 0.0_rk) THEN
         FSTAR           = 0.0_rk
         DFSTAR_DFEFF    = 1.0_rk
      ELSE
C     Critical porosity at different stage of damage evolution
         F_END       = 0.995_rk/Q_1
         FEFF_TOT    = 0.98_rk/Q_1
C     Check material state under damage condition     
         IF (FEFF<F_C_NUM .OR. KAPPA<EPSILON(0.0_rk)) THEN
            FSTAR           = FEFF
            DFSTAR_DFEFF    = 1.0_rk
           IF (FSTAR>FEFF_TOT) THEN
               FZW          = FEFF_TOT
   
               A_FSTAR      = KAPPA/(F_END-FEFF_TOT)
   
               B_FSTAR      = FZW+1.0_rk/A_FSTAR*LOG(1.0_rk-FEFF_TOT/F_END)
   
               FSTAR        = F_END*(1.0_rk-EXP(-A_FSTAR*(FEFF-B_FSTAR)))
   
               DFSTAR_DFEFF = F_END*A_FSTAR*EXP(-A_FSTAR*(FEFF-B_FSTAR))
           END IF
C
         ELSE
            FSTAR           = F_C_NUM+KAPPA*(FEFF-F_C_NUM)
            DFSTAR_DFEFF    = KAPPA
            IF (FSTAR>FEFF_TOT) THEN
               FZW          = (FEFF_TOT-F_C_NUM)/KAPPA+F_C_NUM
   
               A_FSTAR      = KAPPA/(F_END-FEFF_TOT)
   
               B_FSTAR      = FZW+1.0_rk/A_FSTAR*LOG(1.0_rk-FEFF_TOT/F_END)
   
               FSTAR        = F_END*(1.0_rk-EXP(-A_FSTAR*(FEFF-B_FSTAR)))
   
               DFSTAR_DFEFF = F_END*A_FSTAR*EXP(-A_FSTAR*(FEFF-B_FSTAR))
            END IF
         END IF
C
      END IF
C
      RETURN
      END SUBROUTINE damage_evo
C***********************************************************************
C=======================================================================
C***********************************************************************
      subroutine vhetval (
C Read only (unmodifiable) variables -
     1     nblock, nElem, nIntPt, nLayer, nSectPt, 
     2     ntgrad, nstatev, nfieldv, nprops,
     3     cmname, stepTime, totalTime, dt, 
     4     coordMp, density, props,
     5     tempOld, fieldOld, stateOld, 
     6     tempNew, tempgradNew, fieldNew, 
C Write only (modifiable) variables -
     7     stateNew, flux )
C Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C Explicit definition of variables
      IMPLICIT NONE
C     Standard variables for VHETVAL subroutine in ABAQUS
      INTEGER(KIND=intk),INTENT(IN) :: nblock, nElem, nIntPt, nLayer, nSectPt, ntgrad, nstatev, nfieldv, nprops
      CHARACTER(80)::CMNAME
      REAL(KIND=rk),INTENT(IN) :: stepTime, totalTime, dt
      REAL(KIND=rk),INTENT(IN) :: coordMp(nblock,3)
      REAL(KIND=rk),DIMENSION(nblock),INTENT(IN) ::density
      REAL(KIND=rk),DIMENSION(nprops),INTENT(IN) :: props
      REAL(KIND=rk),DIMENSION(nblock),INTENT(IN) :: tempOld, tempNew
      REAL(KIND=rk),DIMENSION(nblock,nfieldv),INTENT(IN) :: fieldOld, fieldNew
      REAL(KIND=rk),DIMENSION(nblock,nstatev),INTENT(IN) :: stateOld
      REAL(KIND=rk),DIMENSION(nblock,ntgrad),INTENT(IN) :: tempgradNew
      
      REAL(KIND=rk),DIMENSION(nblock),INTENT(OUT) :: flux
      REAL(KIND=rk),DIMENSION(nblock,nstatev):: stateNew
C     Local variables declaration
      INTEGER(KIND=intk) :: kblock
C-----------------------------------------------------------------------
      do kblock = 1, nblock
         flux(kblock) = (stateNew(kblock,5)-tempNew(kblock))*stateNew(kblock,7)
      end do 
C
      return
      end
C***********************************************************************
C=======================================================================
C***********************************************************************
C A non-standard utiliy subroutine
C
      SUBROUTINE nonlocalderivative (nblock,nblock_cur,nstatev,stateOld,
     1                  nprops,props,feffNew,fstar,vmisestr,shydtr,
     2                  DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV,DELTA_RR,
     3                  KMATRIX,RMATRIX)
C     Integrate module "Abaqus_Interface" for parameter definition
      USE Abaqus_Interface
C     Use module, which defines a object with the material parameters
C      USE PROP_MODULE
C     Explicit definition of variables
      IMPLICIT NONE
C     Variables declaration for DERIVATIVES_NL Subroutine
C     Input variables
      INTEGER(KIND=intk),INTENT(IN) ::nblock,nblock_cur,nstatev
      REAL(KIND=rk),DIMENSION(nblock,nstatev),INTENT(IN)::stateOld
      INTEGER(KIND=intk),INTENT(IN) ::nprops
      REAL(KIND=rk),DIMENSION(nprops),INTENT(IN)::props
      REAL(KIND=rk),INTENT(IN)      ::feffNew,fstar
      REAL(KIND=rk),INTENT(IN)   ::vmisestr,shydtr      
      REAL(KIND=rk),INTENT(IN)   ::DELTA_LAMBDA_PL,DELTA_LAMBDA_PLV
C     Output variables
      REAL(KIND=rk),INTENT(OUT)                      ::DELTA_RR
      REAL(KIND=rk),DIMENSION(2,2),INTENT(OUT)       ::KMATRIX
      REAL(KIND=rk),DIMENSION(2),INTENT(OUT)         ::RMATRIX
C     Increment of nucleation fraction of void in the model
c!      REAL(KIND=rk),INTENT(OUT)    ::delta_nucleation
C-----------------------------------------------------------------------
C     Local variables
      REAL(KIND=rk) ::EPSILON_MISES,RR
      REAL(KIND=rk) ::R,HH,DFSTAR_DFEFF
      REAL(KIND=rk) ::TAU_MISES,TAU_HYD
      REAL(KIND=rk) ::vmises,shyd
      REAL(KIND=rk) ::DR1_DTAU_MISES,DTAU_MISES_DLAMBDA_PL,DR1_DR,
     1                         DR_DRR,DDELTA_R_DDELTA_LAMBDA_PL,
     5                         DR2_DTAU_MISES,DR2_DTAU_HYD,
     6                         DTAU_HYD_DDELTA_LAMBDA_PL
      REAL(KIND=rk)::COS_H,SIN_H,D2R1_DTAU_MISES2,
     1                         D2R1_DTAU_MISES_DR,D2R1_DTAU_HYD2,
     2                         D2R1_DTAU_HYD_DFSTAR,D2R1_DTAU_HYD_DR,
     3                         DR1_DTAU_HYD,DTAU_HYD_DDELTA_LAMBDA_PLV,
     4                         DR1_DFSTAR

      REAL(KIND=rk)  ::EPS_EQ,DEPS_EQ_DLAMBDA_PL,
     1                         DEPS_EQ_DLAMBDA_PLV,ZUSATZ,ZUSATZ_EPS_EQ,
     2                         ZUSATZ_LAMBDA_PL,ZUSATZ_LAMBDA_PLV,
     3                         ZUSATZ_TAU_MISES,ZUSATZ_TAU_HYD,
     4                         ZUSATZ_DELTA_FEFF
C     Local variables for calling VUHARD
      INTEGER(KIND=intk)::nvalue
      REAL(KIND=rk)     ::hard
C     Nucleation fraction of void in the model
      REAL(KIND=rk)::nucleation
C     Local variable for calculating hardening variable through NR-Loop
      INTEGER(KIND=intk)::NUM_ITER
      REAL(KIND=rk)     ::ERROR
C     Local variable for subroutine STRAIN_HARDENING
      REAL(KIND=rk)     ::SYIELD,HARDMODUL
C      REAL(KIND=rk)     ::Q_1,Q_2
C     Declaration of TYPE, which will contains material PROPS
C      TYPE(PROPERTIES):: MATPARA
C-----------------------------------------------------------------------
C     Intrinsic function
      INTRINSIC MATMUL,ATAN,EPSILON,EXP,LOG,COSH,SINH
C-----------------------------------------------------------------------
C     Explicit number definition
      REAL(KIND=rk),PARAMETER::PI=3.141592653589793_rk
      REAL(KIND=rk),PARAMETER::TOL=1E-10_rk
      REAL(KIND=intk),PARAMETER::MAX_ITER=10_intk 
C-----------------------------------------------------------------------
C-----------------------------------------------------------------------
C     Declaration of variables which take material parameters as input
C     Variables for elastic property of material
      REAL(KIND=rk)   ::E,NU,K,G,LAMBDA
C     Variable for material porosity
      REAL(KIND=rk)   ::F_0,F_C,KAPPA
C     GTN-Model parameter
      REAL(KIND=rk)   ::Q_1,Q_2
C     Variable for void nucleation
      REAL(KIND=rk)   ::F_N,E_n,S_n
C     Variable for choosing options between GTN-Model and Non-local GTN-Model
      REAL(KIND=rk)   ::DAMAGE_CASE
C     Variables for plastic property of material
      REAL(KIND=rk)   ::TAU_0
      REAL(KIND=rk)   ::N
C-----------------------------------------------------------------------
C     Assign material properties and model parameters
C     Young's Modulus
      E               =PROPS(1)
C     Poisson's ratio
      NU              =PROPS(2)
C     Initial porosity
      F_0             =PROPS(3)
C     Critial porosity
      F_C             =PROPS(4)
C     Softening parameter by void evolution
      KAPPA           =PROPS(5)
C     GTN-Model parameter
      Q_1             =PROPS(6)
      Q_2             =PROPS(7)
C     Paramter for IF-condtion between local and non-local GTN-Model
      DAMAGE_CASE     =PROPS(8)
C     Pore nucleation parameter
C     Particle volume fraction
      F_N             =PROPS(9)
C     Fit paramter of 
      S_n               =PROPS(10)
C     Fit paramter of Weibull modulus
      E_n             =PROPS(11)
C     Strain hardening paramters
C     Initial yield stress
      TAU_0           =PROPS(12)
C     Exponent for strain hardening power law
      N               =PROPS(13)
C-----------------------------------------------------------------------
C     Assign solution-dependent state variables to local variables
      RR             =stateOld(nblock_cur,2)
C     Calculate equivalent Mises-stress and hydrostatic stress
      K           = E/(3.0_rk*(1.0_rk-2.0_rk*NU))
      G           = E/(2.0_rk*(1.0_rk+NU))
      vmises = vmisestr -3.0_rk*G*DELTA_LAMBDA_PL
      shyd   = shydtr   -3.0_rk*K*DELTA_LAMBDA_PLV
C     Assign values for variables used in Newton-Raphson-Loop
      EPS_EQ      = DELTA_LAMBDA_PL*0.5_rk 
C     Newton-Raphson-Loop for hardening variable
      NUM_ITER     =0 
      ERROR       =2.0_rk*TOL
      DO WHILE (ERROR>TOL .AND. NUM_ITER<=MAX_ITER)
C                
            DELTA_RR     =EPS_EQ
C
            if (nprops .eq. 13) then
               call hardening_curve (nprops,props,RR + DELTA_RR,syield,hard)
            else
               nvalue = (nprops-13)/2
               call vuhard(syield, hard, RR + DELTA_RR, props(14), nvalue)
            end if
            ZUSATZ  =EPS_EQ*SYIELD-(vmises*DELTA_LAMBDA_PL
     1                   +3.0_rk*shyd*DELTA_LAMBDA_PLV)
     2                   /(1.0_rk-feffNew) 
            DR_DRR                      =hard
            DDELTA_R_DDELTA_LAMBDA_PL   =1.0_rk 
            ZUSATZ_EPS_EQ =SYIELD+EPS_EQ*DR_DRR
            EPS_EQ      =EPS_EQ-ZUSATZ/ZUSATZ_EPS_EQ 
            ERROR       =abs(ZUSATZ)+abs(ZUSATZ/ZUSATZ_EPS_EQ) 
            NUM_ITER     =NUM_ITER+1_intk
      END DO
C     End of Newton-Raphson-Loop for hardening variable
C
C     Derivative of hardening variable
C
      DELTA_RR     =EPS_EQ 
C     Call subroutine to calculate yield stress and hardening modulus
      if (nprops .eq. 13) then
         call hardening_curve (nprops,props,RR + DELTA_RR,syield,hard)
      else
         nvalue = (nprops-13)/2
         call vuhard(syield, hard, RR + DELTA_RR, props(14), nvalue)
      end if
C     Assign yield stressn and hardening modulus to local variables
      R                           =syield                                                                    
      DR_DRR                      =hard 
      DDELTA_R_DDELTA_LAMBDA_PL   =1.0_rk 

C     Derivatives of additional equation for hardening variable
      DTAU_MISES_DLAMBDA_PL       =-3.0_rk*G 
      DTAU_HYD_DDELTA_LAMBDA_PLV  =-3.0_rk*K 
      ZUSATZ_EPS_EQ   =R+EPS_EQ*DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL 
     
      ZUSATZ_LAMBDA_PL    =-1.0_rk/(1.0_rk-feffNew)
     1                   *(vmises+DELTA_LAMBDA_PL
     2                   *DTAU_MISES_DLAMBDA_PL)
                
      ZUSATZ_LAMBDA_PLV    =-1.0_rk/(1.0_rk-feffNew)
     1                   *3.0_rk*(shyd+DELTA_LAMBDA_PLV
     2                   *DTAU_HYD_DDELTA_LAMBDA_PLV)
                            
      DEPS_EQ_DLAMBDA_PL   =-ZUSATZ_LAMBDA_PL/ZUSATZ_EPS_EQ 
                
      DEPS_EQ_DLAMBDA_PLV  =-ZUSATZ_LAMBDA_PLV/ZUSATZ_EPS_EQ 
      
      COS_H        =COSH(Q_2*3.0_rk/2.0_rk*shyd/R)
      SIN_H        =SINH(Q_2*3.0_rk/2.0_rk*shyd/R)

C     Derivatives of yield function
      DR1_DTAU_MISES         =2.0_rk*vmises/R**2
      D2R1_DTAU_MISES2       =2.0_rk/R**2
      D2R1_DTAU_MISES_DR     =-4.0_rk*vmises/R**3 
      D2R1_DTAU_HYD2         =9.0_rk/2.0_rk*Q_1*Q_2**2*FSTAR/R**2*COS_H 
      D2R1_DTAU_HYD_DFSTAR   =3.0_rk*Q_1*Q_2/R*SIN_H 
      D2R1_DTAU_HYD_DR       =-3.0_rk*Q_1*Q_2*FSTAR/R**2*SIN_H
     1                               -9.0_rk*Q_2**2*Q_1*FSTAR*shyd
     2                               /2.0_rk/R**3*COS_H 
      DR1_DTAU_HYD           =3.0_rk*Q_1*Q_2*FSTAR/R*SIN_H 
      
      DR1_DR                 =-2.0_rk*vmises**2/R**3-3.0_rk*Q_1*Q_2
     1                               *FSTAR*shyd/R**2*SIN_H 
      DR1_DFSTAR             =2.0_rk*Q_1*COS_H-2.0_rk*Q_1**2*FSTAR
C     Assemble tangent matrix (Matrix of derivatives for Newton-Raphson)
      KMATRIX       =0.0_rk

      KMATRIX(1,1)  =DR1_DTAU_MISES*DTAU_MISES_DLAMBDA_PL+DR1_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PL

      KMATRIX(1,2)  =DR1_DTAU_HYD*DTAU_HYD_DDELTA_LAMBDA_PLV
                
      KMATRIX(1,2)  =KMATRIX(1,2)+DR1_DR*DR_DRR
     1                   *DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PLV 
                
      KMATRIX(2,1)  =DELTA_LAMBDA_PLV*(D2R1_DTAU_MISES2
     1                   *DTAU_MISES_DLAMBDA_PL+D2R1_DTAU_MISES_DR
     2                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     3                   *DEPS_EQ_DLAMBDA_PL) 
                
      KMATRIX(2,1)  =KMATRIX(2,1)-DR1_DTAU_HYD*1.0_rk/3.0_rk
                
      KMATRIX(2,1)  =KMATRIX(2,1)-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *D2R1_DTAU_HYD_DR*DR_DRR
     2                   *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PL                  
                
      KMATRIX(2,2)  =DR1_DTAU_MISES-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *(D2R1_DTAU_HYD2*DTAU_HYD_DDELTA_LAMBDA_PLV) 
                
      KMATRIX(2,2)  =KMATRIX(2,2)+DELTA_LAMBDA_PLV*D2R1_DTAU_MISES_DR
     1                   *DR_DRR*DDELTA_R_DDELTA_LAMBDA_PL
     2                   *DEPS_EQ_DLAMBDA_PLV 
                
      KMATRIX(2,2)  =KMATRIX(2,2)-DELTA_LAMBDA_PL*1.0_rk/3.0_rk
     1                   *D2R1_DTAU_HYD_DR*DR_DRR
     2                   *DDELTA_R_DDELTA_LAMBDA_PL*DEPS_EQ_DLAMBDA_PLV 

C     Right-hand-side vector R for Newton-Raphson
      RMATRIX      =0.0_rk
      RMATRIX(1)   =vmises**2/R**2+2.0_rk*Q_1*FSTAR*COS_H
     1                         -1.0_rk-Q_1**2*FSTAR**2
      RMATRIX(2)   =DELTA_LAMBDA_PLV*DR1_DTAU_MISES-DELTA_LAMBDA_PL
     1                         *1.0_rk/3.0_rk*DR1_DTAU_HYD
C
      RETURN
      END SUBROUTINE nonlocalderivative
C***********************************************************************
C=======================================================================
C***********************************************************************
      SUBROUTINE MATRIX_INVERSION_2_2 (MATRIX, INVERSEMATRIX)
C
      USE Abaqus_Interface
C
      IMPLICIT NONE
C
C     EINGANGSMATRIX
      REAL(KIND=rk),DIMENSION(2,2),INTENT(IN)   ::MATRIX
C     ERGEBNISMATRIX
      REAL(KIND=rk),DIMENSION(2,2),INTENT(OUT)  ::INVERSEMATRIX 
C     LOCAL VARIABLES
      REAL(KIND=rk)                             ::J_DET
C-----------------------------------------------------------------------
      IF (SIZE(MATRIX) .EQ. 4) THEN
      ELSE
        CALL XPLB_ABQERR(-3,'SIMULATION INTERRUPTED DUE TO'//
     1       ' WRONG MATRIX INPUT. IT SHOULD BE 2X2 MATRIX',1,1.0_rk,'')
      END IF
C
      J_DET = MATRIX(1,1)*MATRIX(2,2)-MATRIX(1,2)*MATRIX(2,1)
C    
      INVERSEMATRIX(1,1) =  MATRIX(2,2)
      INVERSEMATRIX(1,2) = -MATRIX(1,2)
C    
      INVERSEMATRIX(2,1) = -MATRIX(2,1)
      INVERSEMATRIX(2,2) =  MATRIX(1,1)
C    
      INVERSEMATRIX      =  1.0_rk/J_DET*INVERSEMATRIX
C
      RETURN
      END SUBROUTINE MATRIX_INVERSION_2_2
C***********************************************************************
C=======================================================================
C***********************************************************************
