import numpy as np
import matplotlib.pyplot as plt
from matplotlib.animation import FuncAnimation

# Simulation parameters
size = 200       # domain size
dx = dy = 1.0    # spatial discretization
dt = 1.0         # time step
c = 0.2          # wave speed

# Create a spatial grid
x = np.linspace(0, size-1, size)
y = np.linspace(0, size-1, size)
X, Y = np.meshgrid(x, y)

# Initialize fields
u = np.zeros((size, size))   # current wave field
u_prev = np.zeros((size, size))  # previous wave field
u_next = np.zeros((size, size))  # next wave field

# Source parameters
source_amplitude = 80
source_frequency = 0.04
source_position = (100, 100)

def source(t):
    """ Returns the source value at time t """
    return source_amplitude * np.sin(2 * np.pi * source_frequency * t)

def update(u, u_prev, t, boundary_condition='Mur'):
    """ Update wave field for the next time step using the FDTD method """
    # Core update (finite difference)
    for i in range(1, size-1):
        for j in range(1, size-1):
            u_next[i, j] = (c**2 * dt**2 / dx**2) * (u[i+1, j] + u[i-1, j] + u[i, j+1] + u[i, j-1] - 4*u[i, j]) + 2*u[i, j] - u_prev[i, j]

    # Apply source
    u_next[source_position] += source(t)

    # Apply boundary conditions
    if boundary_condition == 'Mur':  # Absorbing boundary
        for j in range(size):
            # Left and right boundaries
            u_next[0, j] = u[1, j] - (dx-c*dt)/(dx+c*dt) * (u_next[1, j] - u[0, j])
            u_next[-1, j] = u[-2, j] - (dx-c*dt)/(dx+c*dt) * (u_next[-2, j] - u[-1, j])
            # Top and bottom boundaries
            u_next[j, 0] = u[j, 1] - (dx-c*dt)/(dx+c*dt) * (u_next[j, 1] - u[j, 0])
            u_next[j, -1] = u[j, -2] - (dx-c*dt)/(dx+c*dt) * (u_next[j, -2] - u[j, -1])
    elif boundary_condition == 'Dirichlet':  # Reflecting boundary
        u_next[0, :] = u_next[-1, :] = u_next[:, 0] = u_next[:, -1] = 0

    # Update arrays for next iteration
    u_prev[:, :] = u[:, :]
    u[:, :] = u_next[:, :]

def animate(i):
    """ Animation function """
    update(u, u_prev, i, boundary_condition='Mur')  # Toggle 'Mur' or 'Dirichlet'
    img.set_data(u)
    img.set_clim(-source_amplitude, source_amplitude)
    return img,

# Setup plot for animation
fig, ax = plt.subplots()
img = ax.imshow(u, origin='lower', cmap='viridis', 
                interpolation='nearest', animated=True, vmin=-source_amplitude, vmax=source_amplitude)
ax.set_title("Wave Propagation in homogeneous medium", fontsize = 12)
ax.set_xlabel("x-axis", fontsize = 12)
ax.set_ylabel("y-axis", fontsize = 12)
cbar = fig.colorbar(img, ax=ax)
cbar.set_label('Wave Amplitude')

# Create animation
ani = FuncAnimation(fig, animate, frames=1000, interval=50, blit=True)

# Optionally save the animation
#ani.save('wave_propagation2.mp4', writer='ffmpeg')
plt.show()
