
#instruction to run the code.
#* line 104 and 106 , allows user to save GIF and animated video of this visualization.
#* it can be best visualized if you run line 104 as GIF ,as Video processing time may take a while.



import numpy as np
import matplotlib.pyplot as plt
import matplotlib.animation as animation
from matplotlib import cm
from mpl_toolkits.mplot3d import Axes3D

# Constants
c0 = 3e8  # Speed of light in vacuum
eps0 = 8.854e-12  # Permittivity of free space
mu0 = 4 * np.pi * 1e-7  # Permeability of free space

# Simulation parameters
dx = 0.01  # Spatial step (m)
dy = 0.01  # Spatial step (m)
dt = 0.9 * dx / (c0 * np.sqrt(2))  # Time step (s)
Lx = 2.0  # Domain length in x (m)
Ly = 2.0  # Domain length in y (m)
T = 1e-7  # Total simulation time (s)

Nx = int(Lx / dx)  # Number of spatial steps in x
Ny = int(Ly / dy)  # Number of spatial steps in y
Nt = int(T / dt)  # Number of time steps

# Material properties for circular dielectric medium
eps_r = np.ones((Nx, Ny))
sigma = np.zeros((Nx, Ny))
center_x = int(0.5* Nx)
center_y = int(0.5 * Ny)
radius = int(0.20 / dx)

for i in range(Nx):
    for j in range(Ny):
        if (i - center_x)**2 + (j - center_y)**2 <= radius**2:
            eps_r[i, j] = 30
            sigma[i, j] = 0.3

# Field arrays
Ez = np.zeros((Nx, Ny))
Hx = np.zeros((Nx, Ny))
Hy = np.zeros((Nx, Ny))

# Source parameters
source_position = (Nx // 4, Ny // 2)
pulse_width = 1e-10

f0 = 5e6 # Frequency (Hz)

# PML boundary conditions
pml_thickness = 20  # Number of cells in PML
sigma_max = 0.1  # Maximum conductivity in PML

def apply_pml(sigma, pml_thickness, sigma_max):
    for i in range(pml_thickness):
        sigma[i, :] = sigma_max * (pml_thickness - i) / pml_thickness
        sigma[-i-1, :] = sigma_max * (pml_thickness - i) / pml_thickness
        sigma[:, i] = sigma_max * (pml_thickness - i) / pml_thickness
        sigma[:, -i-1] = sigma_max * (pml_thickness - i) / pml_thickness

apply_pml(sigma, pml_thickness, sigma_max)

def update_fields(Ez, Hx, Hy, eps_r, sigma):
    # Update magnetic fields Hx and Hy
    Hx[:, :-1] -= (dt / mu0 / dy) * (Ez[:, 1:] - Ez[:, :-1])
    Hy[:-1, :] += (dt / mu0 / dx) * (Ez[1:, :] - Ez[:-1, :])
    
    # Update electric field Ez
    Ez[1:-1, 1:-1] += (dt / eps0 / eps_r[1:-1, 1:-1]) * (
        (Hy[1:-1, 1:-1] - Hy[:-2, 1:-1]) / dy -
        (Hx[1:-1, 1:-1] - Hx[1:-1, :-2]) / dx
    )
    
    # Apply conductivity for absorption
    Ez *= np.exp(-sigma * dt / eps0)

def run_simulation():
    frames = []
    for n in range(Nt):
        t = n * dt
        pulse = np.exp(-((t - 4 * pulse_width) ** 2) / (pulse_width ** 2)) * np.cos(2 * np.pi * f0 * t)
        Ez[source_position] += pulse
        
        update_fields(Ez, Hx, Hy, eps_r, sigma)
        
        if n % 10 == 0:
            frames.append(np.copy(Ez))
    
    return frames
def animate(frames, title):
    fig, ax = plt.subplots()
    
    def update(frame):
        ax.clear()
        im = ax.imshow(frame.T, cmap=cm.viridis, vmin=-0.01, vmax=0.01, animated=True)
        ax.set_title(title)
        return [im]
    
    ani = animation.FuncAnimation(fig, update, frames=frames, interval=50, blit=True)
    #Save as GIF using Pillow
    #ani.save("2D_FDTD_Simulation.gif", writer='pillow')
    #Save as MP4 video using ffmpeg
    #ani.save("2D_FDTD_Simulation.mp4", writer='ffmpeg')
    plt.show()
    return ani

def run_simulation_3D_plot():
    frames_1 = []
    time_frames = [ 100, 500,1000,1200]  # Time frames for capturing the results
    for n in range(Nt):
        t = n * dt
        pulse = np.exp(-((t - 4 * pulse_width) ** 2) / (pulse_width ** 2)) * np.cos(2 * np.pi * f0 * t)
        Ez[source_position] += pulse
        
        update_fields(Ez, Hx, Hy, eps_r, sigma)
        
        if n in time_frames:
            frames_1.append(np.copy(Ez))
    
    return frames_1
def plot_3d_surface(Ez, title):
    fig = plt.figure()
    ax = fig.add_subplot(111, projection='3d')
    X, Y = np.meshgrid(np.arange(Ez.shape[0]), np.arange(Ez.shape[1]))
    surf = ax.plot_surface(X, Y, Ez.T, cmap=cm.viridis,edgecolor='black')
    ax.set_title(title)
    ax.set_xlabel('X')
    ax.set_ylabel('Y')
    ax.set_zlabel('Electric Field (Vm$^{-1}$)')
   
    plt.show()

def calculate_energy(Ez, Hx, Hy, eps_r, dx, dy):
    electric_energy = 0.5 * eps0 * np.sum(eps_r * Ez**2) * dx * dy
    magnetic_energy = 0.5 * np.sum((Hx**2 + Hy**2) / mu0) * dx * dy
    total_energy = electric_energy + magnetic_energy
    return total_energy

def run_simulation_with_energy():
    energies = []
    for n in range(Nt):
        t = n * dt
        pulse = np.exp(-((t - 4 * pulse_width) ** 2) / (pulse_width ** 2)) * np.cos(2 * np.pi * f0 * t)
        Ez[source_position] += pulse
        
        update_fields(Ez, Hx, Hy, eps_r, sigma)
        
        if n % 10 == 0:
            energy = calculate_energy(Ez, Hx, Hy, eps_r, dx, dy)
            energies.append(energy)
    return energies
# Run the simulations and plotting
energies = run_simulation_with_energy()
plt.figure()
plt.plot(np.arange(0, Nt, 10) * dt, energies)
plt.xlabel('Time (s)')
plt.ylabel('Total Electromagnetic Energy (J)')
plt.title('Energy Conservation in 2D FDTD Simulation')
plt.grid(True)
plt.show()
# Run simulation and create animation , create 3D surface plots
frames = run_simulation()
frames_1 = run_simulation_3D_plot()

time_labels = [100 , 500 , 1000 , 1200]
animate(frames, '2D FDTD Simulation of Gaussian Pulse')
for frame_1, label in zip(frames_1, time_labels):
    plot_3d_surface(frame_1, f'2D FDTD Simulation at T={label:} s')
