import numpy as np
import matplotlib.pyplot as plt
from scipy.spatial import Voronoi, voronoi_plot_2d

def generate_strongly_decreasing_points(domain_size, ngrains):
    """
    Generate points for a Poisson Voronoi tessellation with a strongly decreasing density,
    ensuring a noticeable gradient in grain size across the domain.
    """
    points = []
    while len(points) < ngrains:
        x = np.random.uniform(0, domain_size)
        y = np.random.uniform(0, domain_size)
        # Use a non-linear decreasing function for point density
        probability = (domain_size - x)**2 / domain_size**2  # Quadratically decreasing probability
        if np.random.rand() < probability:
            points.append([x, y])
        if len(points) > ngrains:  # Reset if over-generated
            points = []

    return np.array(points[:ngrains])

def initialize_order_parameters_exactly(grid_size, dx, points, ngrains):
    """
    Initialize the order parameters for the phase field method, ensuring the specified number of grains.
    """
    eta = np.zeros((grid_size, grid_size, ngrains))

    for i in range(grid_size):
        for j in range(grid_size):
            pos = np.array([i * dx, j * dy])
            distances = np.linalg.norm(points - pos, axis=1)
            closest_point_index = np.argmin(distances)
            eta[i, j, closest_point_index] = 1.0

    # Ensure eta array reflects exactly 'ngrains' grains
    assert eta.shape[2] == ngrains, "The number of grains in eta does not match ngrains."

    return eta

# Parameters remain the same as the previous example
domain_size = 32
ngrains = 25
grid_size = 64
dx = dy = 0.5

# Generate points with a strongly decreasing process
points_strong_gradient = generate_strongly_decreasing_points(domain_size, ngrains)

# Create the Voronoi tessellation
voronoi_strong_gradient = Voronoi(points_strong_gradient)

# Initialize the order parameters
eta_strong_gradient = initialize_order_parameters_exactly(grid_size, dx, points_strong_gradient, ngrains)

# Plot the Voronoi tessellation with a strong gradient
fig, ax = plt.subplots()
voronoi_plot_2d(voronoi_strong_gradient, ax=ax, show_vertices=False)
ax.set_xlim(0, domain_size)
ax.set_ylim(0, domain_size)
ax.set_title("Poisson Voronoi Tessellation with Strong Gradient")
plt.show()

