import numpy as np
import matplotlib.pyplot as plt

def initialize_ideal_grain_growth(Nx, Ny, dx, dy, radius_factor, ngrains=2):
    """
    Initialize the order parameters for ideal grain growth with one circular grain
    embedded in a larger grain.
    Parameters:
    - Nx, Ny: Size of the rectangular grid.
    - dx, dy: Grid spacing.
    - radius_factor: Factor to determine the radius of the circular grain.
    - ngrains: Number of grains, default is 2.
    Returns:
    - eta: Initialized order parameters for each grain in the form of an array
           of shape (Nx, Ny, ngrains).
    """
    # Initialize the order parameter array
    eta = np.zeros((Nx, Ny, ngrains))
    # Calculate the center of the grid
    center_x, center_y = Nx / 2, Ny / 2
    # Calculate the radius of the circular grain
    radius = radius_factor * dx

    for x in range(Nx):
        for y in range(Ny):
            # Calculate the distance of each point from the center
            distance = np.sqrt((x - center_x)**2 + (y - center_y)**2)
            # Initialize grain 1 (the larger grain)
            if distance <= radius:
                eta[x, y, 0] = 0  # Inside the circular grain, set grain 1 to 0
                eta[x, y, 1] = 1  # Set grain 2 to 1
            else:
                eta[x, y, 0] = 1  # Outside the circular grain, set grain 1 to 1
    return eta

# Parameters for initialization
Nx, Ny = 64, 64  # Grid size
dx, dy = 0.5, 0.5  # Grid spacing
radius_factor = 28  # Multiplier for the radius based on dx

# Initialize the order parameters
eta_ideal_gg = initialize_ideal_grain_growth(Nx, Ny, dx, dy, radius_factor)

# Visualization
fig, axes = plt.subplots(1, 2, figsize=(12, 6))
titles = ['Grain 1 (Large Grain)', 'Grain 2 (Embedded Circular Grain)']
for i in range(2):
    im = axes[i].imshow(eta_ideal_gg[:, :, i], cmap='viridis', origin='lower')
    axes[i].set_title(titles[i])
    axes[i].axis('off')
    plt.colorbar(im, ax=axes[i])
plt.tight_layout()
plt.show()
