import numpy as np
import matplotlib.pyplot as plt
from scipy.spatial import Voronoi, voronoi_plot_2d

def generate_voronoi(domain_size, ngrains):
    """
    Generate a Voronoi tessellation for a given domain size and number of grains.
    
    Parameters:
    - domain_size: The size of the domain.
    - ngrains: The number of grains to generate.
    
    Returns:
    - Voronoi tessellation object.
    """
    # Generate random points within the domain
    points = np.random.rand(ngrains, 2) * domain_size
    # Create the Voronoi tessellation
    vor = Voronoi(points)
    return vor

def plot_voronoi(vor, domain_size):
    """
    Plot a Voronoi tessellation, extending cells to the domain boundaries.
    
    Parameters:
    - vor: Voronoi tessellation object.
    - domain_size: The size of the domain.
    """
    fig, ax = plt.subplots()
    voronoi_plot_2d(vor, ax=ax, show_vertices=False, show_points=False)
    ax.set_xlim(0, domain_size)
    ax.set_ylim(0, domain_size)
    ax.set_title('Voronoi Tessellation with Extended Cells')
    plt.show()

def initialize_order_parameters(vor, grid_size, dx, ngrains):
    """
    Initialize the order parameters for the phase field method based on Voronoi tessellation.
    
    Parameters:
    - vor: Voronoi tessellation object.
    - grid_size: The size of the grid.
    - dx: The grid spacing.
    - ngrains: The number of grains.
    
    Returns:
    - Initialized order parameters array.
    """
    eta = np.zeros((grid_size, grid_size, ngrains))
    for i in range(grid_size):
        for j in range(grid_size):
            # Find the closest Voronoi cell for each grid point
            x, y = (i + 0.5) * dx, (j + 0.5) * dx
            distances = np.sqrt((vor.points[:, 0] - x)**2 + (vor.points[:, 1] - y)**2)
            closest = np.argmin(distances)
            eta[i, j, closest] = 1.0
    return eta

# Parameters
domain_size = 32
ngrains = 25
grid_size = 64
dx = 0.5

# Generate and plot Voronoi tessellation
vor = generate_voronoi(domain_size, ngrains)
plot_voronoi(vor, domain_size)

# Initialize order parameters based on the Voronoi tessellation
eta = initialize_order_parameters(vor, grid_size, dx, ngrains)

# eta contains the order parameters array of dimensions (64, 64, 25)
