import numpy as np

def calculate_edge_displacement(x1, x2, b, nu):
    theta = np.arctan2(x2, x1)
    r2 = x1**2 + x2**2  # Square of the radial distance r
    u1 = (b / (2 * np.pi)) * (theta + (x1 * x2) / (2 * (1 - nu) * r2))
    u2 = (-b / (2 * np.pi)) * ((1 - 2 * nu) * np.log(r2) / (4 * (1 - nu)) + (x1**2 - x2**2) / (4 * (1 - nu) * r2))
    return u1, u2

def calculate_screw_displacement(x1, x2, b):
    theta = np.arctan2(x2, x1)
    u3 = b * theta / (2 * np.pi)
    return u3

def apply_displacement(atom_positions, dislocation_position, b, nu, direction, dislocation_type):
    modified_atoms = []
    for atom in atom_positions:
        element, x, y, z = atom
        if direction == 'z':
            dx, dy = x - dislocation_position[0], y - dislocation_position[1]
        elif direction == 'x':
            dy, dz = y - dislocation_position[0], z - dislocation_position[1]
        elif direction == 'y':
            dx, dz = x - dislocation_position[0], z - dislocation_position[1]

        if dislocation_type == 'edge':
            if direction == 'z':
                u1, u2 = calculate_edge_displacement(dx, dy, b, nu)
                modified_atoms.append([element, x + u1, y + u2, z])
            elif direction == 'x':
                u1, u2 = calculate_edge_displacement(dy, dz, b, nu)
                modified_atoms.append([element, x, y + u1, z + u2])
            elif direction == 'y':
                u1, u2 = calculate_edge_displacement(dx, dz, b, nu)
                modified_atoms.append([element, x + u1, y, z + u2])
        elif dislocation_type == 'screw':
            u3 = calculate_screw_displacement(dx, dy, b)
            if direction == 'z':
                modified_atoms.append([element, x, y, z + u3])
            elif direction == 'x':
                modified_atoms.append([element, x + u3, y, z])
            elif direction == 'y':
                modified_atoms.append([element, x, y + u3, z])
    return modified_atoms

def insert_dislocation(input_file_name, output_file_name, lattice_constant, dislocation_position, direction, dislocation_type, nu):
    with open(input_file_name, 'r') as file:
        lines = file.readlines()
    
    atoms = [[line.split()[0], float(line.split()[1]), float(line.split()[2]), float(line.split()[3])] for line in lines[2:]]
    
    b = np.sqrt(2) * lattice_constant / 2  # Burgers vector for FCC
    
    modified_atoms = apply_displacement(atoms, dislocation_position, b, nu, direction, dislocation_type)
    
    with open(output_file_name, 'w') as file:
        file.write(f"{len(modified_atoms)}\n")
        file.write(f"Atoms with {dislocation_type} dislocation\n")
        for atom in modified_atoms:
            file.write(" ".join(map(str, atom)) + "\n")

# User input handling
input_file_name = input("Enter the name of the input .xyz file: ")
output_file_name = input("Enter the name of the output .xyz file: ")
lattice_constant = float(input("Enter the lattice constant: "))
nu = float(input("Enter the Poisson's ratio: "))
direction = input("Enter the direction of the dislocation line (x, y, or z): ").lower()
dislocation_type = input("Enter the type of dislocation (screw or edge): ").lower()

# Request additional coordinates based on the direction of the dislocation line
if direction == 'x':
    dislocation_y = float(input("Enter the y-coordinate of the dislocation line: "))
    dislocation_z = float(input("Enter the z-coordinate of the dislocation line: "))
    dislocation_position = (dislocation_y, dislocation_z)
elif direction == 'y':
    dislocation_x = float(input("Enter the x-coordinate of the dislocation line: "))
    dislocation_z = float(input("Enter the z-coordinate of the dislocation line: "))
    dislocation_position = (dislocation_x, dislocation_z)
elif direction == 'z':
    dislocation_x = float(input("Enter the x-coordinate of the dislocation line: "))
    dislocation_y = float(input("Enter the y-coordinate of the dislocation line: "))
    dislocation_position = (dislocation_x, dislocation_y)
else:
    
    
    raise ValueError("Invalid direction. Pleas enter x, y, or z.")

    
insert_dislocation(input_file_name, output_file_name, lattice_constant, dislocation_position, direction, dislocation_type, nu)

