#!/usr/bin/env python
# coding: utf-8

# In[18]:


import numpy as np

# Rotation Matrix Calculation from Miller Indices
def rotation_matrix_from_miller_indices(hkl_x, hkl_y, hkl_z):
    # Normalize input vectors
    x = np.array(hkl_x) / np.linalg.norm(hkl_x)
    y = np.array(hkl_y) / np.linalg.norm(hkl_y)
    z = np.array(hkl_z) / np.linalg.norm(hkl_z)
    
    # Ensure orthogonality and compute the rotation matrix
    if np.dot(x, y) == 0 and np.dot(x, z) == 0 and np.dot(y, z) == 0:
        return np.array([x, y, z]).T
    else:
        raise ValueError("Input Miller indices are not orthogonal.")

# Generate the crystal structure
def generate_crystal_structure(crystal_type, a, c, periodicity_length, orientation):
    rot_matrix = rotation_matrix_from_miller_indices(*orientation)
    positions = []

    if crystal_type == 'FCC':
        primitive_vectors = np.array([[0.5*a, 0.5*a, 0], [0.5*a, 0, 0.5*a], [0, 0.5*a, 0.5*a]])
        basis_vectors = np.array([[0, 0, 0]])
    elif crystal_type == 'BCC':
        primitive_vectors = np.array([[-0.5*a, 0.5*a, 0.5*a], [0.5*a, -0.5*a, 0.5*a], [0.5*a, 0.5*a, -0.5*a]])
        basis_vectors = np.array([[0, 0, 0]])
    elif crystal_type == 'HCP':
        primitive_vectors = np.array([[0.5*a, -0.5*(3**(1/2))*a, 0], [0.5*a, 0.5*(3**(1/2))*a, 0], [0, 0, c]])
        basis_vectors = np.array([[0.5*a, 0.5*(3**(-1/2))*a, 0.25*c], [0.5*a, -0.5*(3**(-1/2))*a, 0.75*c]])
    else:
        raise ValueError("Invalid crystal type.")

    # Apply rotation to primitive and basis vectors
    primitive_vectors = np.dot(primitive_vectors, rot_matrix)
    basis_vectors = np.dot(basis_vectors, rot_matrix)

    # Generate atom positions
    for i in range(-periodicity_length[0], periodicity_length[0] + 1):
        for j in range(-periodicity_length[1], periodicity_length[1] + 1):
            for k in range(-periodicity_length[2], periodicity_length[2] + 1):
                cell_origin = i * primitive_vectors[0] + j * primitive_vectors[1] + k * primitive_vectors[2]
                for basis in basis_vectors:
                    pos = cell_origin + basis
                    if all(0 <= pos[n] <= (periodicity_length[n] + 1) for n in range(3)):
                        positions.append(pos)

    return positions

# Save positions to XYZ file
def save_to_xyz_file(atom_positions):
    with open("Crystal_structure.xyz", "w") as file:
        file.write(f"{len(atom_positions)}\n\n")
        for pos in atom_positions:
            file.write(f"Atom {pos[0]} {pos[1]} {pos[2]}\n")

# User Interface
def main():
    crystal_type, a, c, periodicity_length, orientation = get_user_input()
    positions = generate_crystal_structure(crystal_type, a, c, periodicity_length, orientation)
    save_to_xyz_file(positions)
    print("Crystal structure generated and saved to Crystal_structure.xyz.")

def get_user_input():
    # Collect and validate crystal type
    crystal_type = input("Enter crystal type (FCC, BCC, HCP): ").strip().upper()
    while crystal_type not in ['FCC', 'BCC', 'HCP']:
        print("Invalid crystal type. Please enter FCC, BCC, or HCP.")
        crystal_type = input("Enter crystal type (FCC, BCC, HCP): ").strip().upper()
    
    # Collect and validate lattice constant 'a'
    a = float(input("Enter lattice constant 'a': "))
    
    # Initialize 'c' for HCP and validate if needed
    c = 0
    if crystal_type == 'HCP':
        c = float(input("Enter additional lattice constant 'c' (must be > a): "))
        while c <= a:
            print("Invalid 'c' value. It must be greater than 'a'.")
            c = float(input("Enter additional lattice constant 'c': "))
    
    # Collect and validate periodicity length (x, y, z)
    periodicity_length_str = input("Enter periodicity length (x,y,z): ")
    x, y, z = map(int, periodicity_length_str.split(','))
    
    # Collect and validate orientation
    print("Enter orientation as Miller indices (h,k,l) for x, y, z directions:")
    hkl_x = tuple(map(int, input("Orientation for x direction: ").split(',')))
    hkl_y = tuple(map(int, input("Orientation for y direction: ").split(',')))
    hkl_z = tuple(map(int, input("Orientation for z direction: ").split(',')))
    
    # Ensure orientation vectors are orthogonal
    # Note: This is a simplified check and may need refinement based on specific requirements
    if not (np.dot(hkl_x, hkl_y) == 0 and np.dot(hkl_x, hkl_z) == 0 and np.dot(hkl_y, hkl_z) == 0):
        print("Warning: The provided orientation vectors are not orthogonal. Please check your input.")
    
    return crystal_type, a, c, (x, y, z), (hkl_x, hkl_y, hkl_z)
    
    
if __name__ == "__main__":
    
    main()
    

