import numpy as np

def generate_crystal_structure(crystal_type, a, c, num_cells):
    if crystal_type not in ['FCC', 'BCC', 'HCP']:
        raise ValueError("Invalid crystal type. Must be 'FCC', 'BCC', or 'HCP'.")
    if a <= 0 or (crystal_type == 'HCP' and c <= a):
        raise ValueError("Invalid lattice constants. Ensure 'a' > 0 and for HCP, 'c' > 'a'.")
    if num_cells <= 0:
        raise ValueError("Number of unit cells must be a positive integer.")
    
    # Box length calculation
    if crystal_type == 'HCP':
        box_length_x_y = np.floor(num_cells * a)
        box_height_z = np.floor(num_cells * c)
    else:
        box_length = np.floor(num_cells * a)
    
    atom_positions = []
    if crystal_type in ['FCC', 'BCC']:
        primitive_vectors = np.array([
            [0.5*a, 0.5*a, 0] if crystal_type == 'FCC' else [-0.5*a, 0.5*a, 0.5*a],
            [0.5*a, 0, 0.5*a] if crystal_type == 'FCC' else [0.5*a, -0.5*a, 0.5*a],
            [0, 0.5*a, 0.5*a] if crystal_type == 'FCC' else [0.5*a, 0.5*a, -0.5*a]
        ])
        basis_vectors = np.array([[0, 0, 0]])
        loop_range = range(int(-box_length), int(box_length) + 1)
        for i in loop_range:
            for j in loop_range:
                for k in loop_range:
                    for basis in basis_vectors:
                        pos = i*primitive_vectors[0] + j*primitive_vectors[1] + k*primitive_vectors[2] + basis
                        if all(-box_length <= p <= box_length for p in pos):
                            atom_positions.append(pos)
    elif crystal_type == 'HCP':
        primitive_vectors = np.array([
            [0.5*a, -0.5*np.sqrt(3)*a, 0],
            [0.5*a, 0.5*np.sqrt(3)*a, 0],
            [0, 0, c]
        ])
        basis_vectors = np.array([
            [0.5*a, 0.5/np.sqrt(3)*a, 0.25*c],
            [0.5*a, -0.5/np.sqrt(3)*a, 0.75*c]
        ])
        loop_range_x_y = range(int(-box_length_x_y), int(box_length_x_y) + 1)
        loop_range_z = range(int(-box_height_z), int(box_height_z) + 1)
        for i in loop_range_x_y:
            for j in loop_range_x_y:
                for k in loop_range_z:
                    for basis in basis_vectors:
                        pos = i*primitive_vectors[0] + j*primitive_vectors[1] + k*primitive_vectors[2] + basis
                        if all(-box_length_x_y <= p <= box_length_x_y for p in pos[:2]) and -box_height_z <= pos[2] <= box_height_z:
                            atom_positions.append(pos)

    # Save to XYZ file
    with open(f"{crystal_type}_structure.xyz", 'w') as file:
        file.write(f"{len(atom_positions)}\n")
        file.write(f"{crystal_type} crystal structure\n")
        for pos in atom_positions:
            file.write(f"X {pos[0]} {pos[1]} {pos[2]}\n")

    print(f"{crystal_type} crystal structure generated with {len(atom_positions)} atoms.")

# User input prompts
crystal_type = input("Enter crystal type (FCC, BCC, HCP): ").strip().upper()
a = float(input("Enter lattice constant 'a': "))
c = 0
if crystal_type == 'HCP':
    c = float(input("Enter additional lattice constant 'c' (ensure c > a): "))
num_cells = int(input("Enter number of unit cells in each direction: "))

# Validate inputs
if crystal_type == 'HCP' and c <= a:
    print("Error: For HCP, 'c' must be greater than 'a'.")
else:
    try:
        generate_crystal_structure(crystal_type, a, c, num_cells)
    except ValueError as e:
        print(f"Error: {e}")





