import numpy as np

def generate_crystal_structure(crystal_type, a, c, number_of_unit_cells):
    # Validate crystal type
    if crystal_type not in ['FCC', 'BCC', 'HCP']:
        raise ValueError("Crystal type must be 'FCC', 'BCC', or 'HCP'.")
    
    # Validate lattice constants
    if not a > 0 or (crystal_type == 'HCP' and not c > a):
        raise ValueError("Invalid lattice constants. Ensure 'a' > 0 and for HCP, 'c' > 'a'.")
    
    # Define primitive vectors
    if crystal_type in ['FCC', 'BCC']:
        primitive_vectors = np.array([[0.5*a, 0.5*a, 0], [0.5*a, 0, 0.5*a], [0, 0.5*a, 0.5*a]])
    elif crystal_type == 'HCP':
        primitive_vectors = np.array([[0.5*a, -0.5*np.sqrt(3)*a, 0], [0.5*a, 0.5*np.sqrt(3)*a, 0], [0, 0, c]])
    
    # Calculate periodicity
    norm_primitive = np.linalg.norm(primitive_vectors, axis=1)
    if crystal_type in ['FCC', 'BCC']:
        periodicity_length = (np.array(number_of_unit_cells) * a * norm_primitive).astype(int)
    elif crystal_type == 'HCP':
        periodicity_length_xy = (np.array(number_of_unit_cells[:2]) * a * norm_primitive[:2]).astype(int)
        periodicity_length_z = (number_of_unit_cells[2] * c * norm_primitive[2]).astype(int)
        periodicity_length = np.concatenate([periodicity_length_xy, [periodicity_length_z]])

    return periodicity_length
def get_user_inputs():
    while True:
        crystal_type = input("Enter crystal type (FCC, BCC, HCP): ").upper()
        if crystal_type in ['FCC', 'BCC', 'HCP']:
            break
        print("Invalid crystal type. Please choose from 'FCC', 'BCC', or 'HCP'.")

    a = float(input("Enter lattice constant 'a': "))
    
    if crystal_type == 'HCP':
        c = float(input("Enter additional lattice constant 'c' (must be greater than 'a'): "))
    else:
        c = None  # Not required for FCC and BCC
    
    number_of_unit_cells = input("Enter the number of unit cells in each direction for the simulation box (x,y,z): ")
    number_of_unit_cells = tuple(map(int, number_of_unit_cells.split(',')))
    
    return crystal_type, a, c, number_of_unit_cells

if __name__ == "__main__":
    try:
        crystal_type, a, c, number_of_unit_cells = get_user_inputs()
        periodicity_length = generate_crystal_structure(crystal_type, a, c, number_of_unit_cells)
        print("Periodicity lengths (x, y, z):", periodicity_length)
    except ValueError as e:
        print("Error:", e)

