import numpy as np
import matplotlib.pyplot as plt
import matplotlib.animation as animation
from PIL import Image
import os

# Check for OpenCV availability
try:
    import cv2
    OPENCV_AVAILABLE = True
except ImportError:
    OPENCV_AVAILABLE = False

# Fixed Numerical Parameters
plate_length = 50.0
max_iter_time = 750
alpha = 2.0
delta_x = 1.0
delta_t = (delta_x ** 2) / (4 * alpha)  # Stability criterion
gamma = alpha * delta_t / (delta_x ** 2)

# Boundary Conditions
boundaries = {'top': 100.0, 'bottom': 0.0, 'left': 0.0, 'right': 0.0}

# Hole Configuration
holes = [{'center': (25, 25), 'radius': 5, 'type': 'D', 'value': 0.0}]

def initialize_grid():
    """Initializes the temperature grid with boundary conditions."""
    grid = np.zeros((int(plate_length), int(plate_length)))
    grid[0, :] = boundaries['top']
    # Neumann conditions on other sides are implicitly set to zero
    return grid

def apply_hole_conditions(grid):
    """Applies hole configurations to the grid."""
    for hole in holes:
        x_center, y_center = hole['center']
        radius = hole['radius']
        for i in range(grid.shape[0]):
            for j in range(grid.shape[1]):
                if (i - x_center) ** 2 + (j - y_center) ** 2 < radius ** 2:
                    grid[i, j] = hole['value']
    return grid

def FTCS(grid):
    """Performs a single time step of the FTCS method."""
    new_grid = np.copy(grid)
    for i in range(1, grid.shape[0] - 1):
        for j in range(1, grid.shape[1] - 1):
            new_grid[i, j] = grid[i, j] + gamma * (grid[i+1, j] + grid[i-1, j] + grid[i, j+1] + grid[i, j-1] - 4 * grid[i, j])
    return apply_hole_conditions(new_grid)

def save_frame(frame, iter_time, frames_dir):
    """Saves a single frame of the simulation."""
    plt.imshow(frame, cmap='hot', interpolation='nearest')
    plt.title(f'Time Step: {iter_time}')
    plt.colorbar()
    plt.savefig(f"{frames_dir}/frame_{iter_time:04d}.png")
    plt.close()

def compile_video(frames_dir, output_file):
    """Compiles frames into a video."""
    if not OPENCV_AVAILABLE:
        return
    images = [img for img in os.listdir(frames_dir) if img.endswith(".png")]
    frame = cv2.imread(os.path.join(frames_dir, images[0]))
    height, width, layers = frame.shape
    video = cv2.VideoWriter(output_file, cv2.VideoWriter_fourcc(*'DIVX'), 15, (width, height))
    for image in images:
        video.write(cv2.imread(os.path.join(frames_dir, image)))
    cv2.destroyAllWindows()
    video.release()

def compile_gif(frames_dir, output_file):
    """Compiles frames into a GIF."""
    images = [Image.open(os.path.join(frames_dir, img)) for img in os.listdir(frames_dir) if img.endswith(".png")]
    images[0].save(output_file, save_all=True, append_images=images[1:], optimize=False, duration=40, loop=0)

def run_simulation():
    """Runs the entire simulation."""
    grid = initialize_grid()
    grid = apply_hole_conditions(grid)
    frames_dir = 'frames'
    os.makedirs(frames_dir, exist_ok=True)

    for iter_time in range(max_iter_time):
        print(f"Iteration {iter_time}/{max_iter_time}")
        save_frame(grid, iter_time, frames_dir)
        grid = FTCS(grid)

    if OPENCV_AVAILABLE:
        compile_video(frames_dir, 'heat_conduction_simulation.avi')
    else:
        compile_gif(frames_dir, 'heat_conduction_simulation.gif')

if __name__ == "__main__":
    run_simulation()
