import numpy as np

# Corrected apply_reaction_kinetics function with deterministic outcome for testing
def apply_reaction_kinetics(C, state, reaction_threshold, corrosion_probability_top, corrosion_probability_low, step):
    new_state = state.copy()
    if step < 50:
        print(f"Step {step}: No corrosion applied (initial delay).")
        return new_state  # No corrosion until step 50

    corroded = (C >= reaction_threshold) & (state == 0)
    is_top_metal = np.zeros_like(C, dtype=bool)
    is_top_metal[:C.shape[0] // 2, :] = True  # First half rows are top metal
    corrosion_probability = np.where(is_top_metal, corrosion_probability_top, corrosion_probability_low)

    # Set a fixed random seed for reproducibility
    np.random.seed(0)
    probabilistic_corrosion = np.random.rand(*C.shape)
    # Force the cells [4, 5] and [6, 5] to corrode for testing purposes
    probabilistic_corrosion[4, 5] = 0
    probabilistic_corrosion[6, 5] = 0

    for i in range(1, state.shape[0] - 1):
        for j in range(1, state.shape[1] - 1):
            if corroded[i, j]:
                neighbors = state[i - 1:i + 2, j - 1:j + 2]
                if np.any(neighbors == 1):
                    if probabilistic_corrosion[i, j] < corrosion_probability[i, j]:
                        new_state[i, j] = 1  # New cells become corroded with specified probability
                        print(
                            f"Cell ({i}, {j}) corroded. Neighbors: {neighbors.flatten()}, Prob: {corrosion_probability[i, j]}, Rand: {probabilistic_corrosion[i, j]}")
                    else:
                        print(
                            f"Cell ({i}, {j}) not corroded. Neighbors: {neighbors.flatten()}, Prob: {corrosion_probability[i, j]}, Rand: {probabilistic_corrosion[i, j]}")
                else:
                    print(f"Cell ({i}, {j}) has no corroded neighbors.")
            else:
                print(f"Cell ({i}, {j}) not above reaction threshold.")
    return new_state

# Test function for reaction kinetics
# The test function initializes the grid and verifies the corrosion process.
def test_reaction_kinetics():
    N_x, N_y = 10, 10  # Small grid for testing
    C = np.zeros((N_x, N_y))
    state = np.zeros((N_x, N_y))
    # Set initial conditions
    reaction_threshold = 0.1
    corrosion_probability_top = 0.5
    corrosion_probability_low = 0.3
    # Set specific cells to meet the corrosion threshold
    C[4, 5] = 0.2  # Above threshold in top metal
    C[6, 5] = 0.2  # Above threshold in bottom metal
    state[4, 4] = 1  # Initial corroded cell near top threshold cell
    state[6, 6] = 1  # Initial corroded cell near bottom threshold cell

    # Print initial state and concentration
    print("Initial State:")
    print(state)
    print("Initial Concentration (C):")
    print(C)

    # Apply reaction kinetics
    step = 51  # Step beyond the initial delay
    new_state = apply_reaction_kinetics(C, state, reaction_threshold, corrosion_probability_top, corrosion_probability_low, step)

    # Print new state after applying reaction kinetics
    print("New State After Reaction Kinetics:")
    print(new_state)

    # Check if the specific cells have corroded based on their neighbors and probabilities
    assert new_state[4, 5] == 1, "Top metal cell should be corroded."
    assert new_state[6, 5] == 1, "Bottom metal cell should be corroded."

# Run the test
test_reaction_kinetics()
